// JeetBlock Background Script - Weekly Authentication Service
class JeetBlockBackground {
  constructor() {
    this.licenseCheckUrl = 'https://auth.jeetblock.org';
    this.weeklyInterval = null;
    this.retryAttempts = 3;
    this.retryDelay = 30000; // 30 seconds
    this.lastSuccessfulCheck = null;
    
    this.init();
  }

  parseBoolean(value, defaultValue = false) {
    if (typeof value === 'boolean') {
      return value;
    }

    if (typeof value === 'string') {
      const normalized = value.trim().toLowerCase();
      if (normalized === 'true' || normalized === '1') {
        return true;
      }
      if (normalized === 'false' || normalized === '0' || normalized === '') {
        return false;
      }
    }

    if (typeof value === 'number') {
      if (value === 1) {
        return true;
      }
      if (value === 0) {
        return false;
      }
    }

    if (value === null || value === undefined) {
      return defaultValue;
    }

    return Boolean(value);
  }

  async init() {
    // Load last successful check time
    const result = await chrome.storage.local.get(['jeetblock_last_auth_check']);
    this.lastSuccessfulCheck = result.jeetblock_last_auth_check || 0;
    
    // Start weekly authentication ping
    this.startWeeklyAuthPing();
    
    // Icon click will open popup (handled by manifest)
    
    // Initialize icon based on current state
    await this.initializeIcon();
    
    // Listen for messages from popup/content scripts
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      this.handleMessage(request, sender, sendResponse);
      return true; // Keep message channel open for async response
    });
  }

  startWeeklyAuthPing() {
    // Weekly ping for authentication - 7 days = 604,800,000 ms
    this.weeklyInterval = setInterval(async () => {
      await this.performWeeklyAuthCheck();
    }, 7 * 24 * 60 * 60 * 1000); // Weekly ping for authentication - 604,800,000 ms = 7 days
    
    // Also perform initial check if it's been more than a week since last check
    const now = Date.now();
    const weekInMs = 7 * 24 * 60 * 60 * 1000;
    
    if (!this.lastSuccessfulCheck || (now - this.lastSuccessfulCheck) >= weekInMs) {
      // Perform immediate check if it's been a week or more
      this.performWeeklyAuthCheck();
    }
  }

  async performWeeklyAuthCheck() {
    console.log('JeetBlock: Performing weekly authentication check...');
    
    try {
      const result = await this.checkLicenseWithRetry();
      
      if (result.success) {
        this.lastSuccessfulCheck = Date.now();
        await chrome.storage.local.set({
          'jeetblock_last_auth_check': this.lastSuccessfulCheck,
          'jeetblock_auth_status': 'valid'
        });
        console.log('JeetBlock: Weekly authentication check successful');
      } else {
        console.warn('JeetBlock: Weekly authentication check failed:', result.error);
        await chrome.storage.local.set({
          'jeetblock_auth_status': 'failed',
          'jeetblock_auth_error': result.error
        });
      }
    } catch (error) {
      console.error('JeetBlock: Weekly authentication check error:', error);
      await chrome.storage.local.set({
        'jeetblock_auth_status': 'error',
        'jeetblock_auth_error': error.message
      });
    }
  }

  async checkLicenseWithRetry() {
    let lastError = null;
    
    for (let attempt = 1; attempt <= this.retryAttempts; attempt++) {
      try {
        console.log(`JeetBlock: Authentication attempt ${attempt}/${this.retryAttempts}`);
        
        const result = await this.checkLicense();
        
        if (result.licensed) {
          return { success: true, licensed: true, licenseKey: result.licenseKey };
        } else {
          lastError = new Error(result.error || 'License check failed');
        }
      } catch (error) {
        lastError = error;
        console.warn(`JeetBlock: Authentication attempt ${attempt} failed:`, error.message);
      }
      
      // Wait before retry (except on last attempt)
      if (attempt < this.retryAttempts) {
        await new Promise(resolve => setTimeout(resolve, this.retryDelay));
      }
    }
    
    return { success: false, error: lastError?.message || 'All retry attempts failed' };
  }

  async checkLicense() {
    const deviceId = await this.getDeviceId();
    
    const response = await fetch(this.licenseCheckUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        deviceId: deviceId
      })
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    
    return {
      licensed: data.licensed || false,
      licenseKey: data.licenseKey || null,
      error: data.error || null
    };
  }

  async getDeviceId() {
    // Try to get existing device ID from chrome.storage.local first
    const result = await chrome.storage.local.get(['jeetblock_device_id']);
    
    if (result.jeetblock_device_id) {
      return result.jeetblock_device_id;
    }
    
    // Generate new device ID if none exists
    const deviceId = await this.generateDeviceId();
    await chrome.storage.local.set({ 'jeetblock_device_id': deviceId });
    return deviceId;
  }

  async generateDeviceId() {
    try {
      // Create a fingerprint from browser and device characteristics
      const fingerprint = await this.createFingerprint();
      const deviceId = `JEETBLOCK-${fingerprint}`;
      return deviceId;
    } catch (error) {
      console.error('Error generating device ID:', error);
      // Fallback to a simple ID if fingerprinting fails
      return `JEETBLOCK-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    }
  }

  async createFingerprint() {
    // Note: In background script, we have limited access to browser APIs
    // We'll use a simpler fingerprinting approach
    const components = [
      navigator.userAgent,
      navigator.language,
      new Date().getTimezoneOffset().toString(),
      navigator.hardwareConcurrency?.toString() || 'unknown',
      navigator.platform
    ];

    // Create a hash from the components
    const fingerprintString = components.join('|');
    const encoder = new TextEncoder();
    const data = encoder.encode(fingerprintString);
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    
    return hashHex.substring(0, 16).toUpperCase(); // Return first 16 characters
  }

  async handleMessage(request, sender, sendResponse) {
    try {
      switch (request.action) {
        case 'getAuthStatus':
          const authResult = await chrome.storage.local.get([
            'jeetblock_auth_status',
            'jeetblock_last_auth_check',
            'jeetblock_auth_error'
          ]);
          sendResponse({
            status: authResult.jeetblock_auth_status || 'unknown',
            lastCheck: authResult.jeetblock_last_auth_check || 0,
            error: authResult.jeetblock_auth_error || null
          });
          break;
          
        case 'forceAuthCheck':
          const result = await this.performWeeklyAuthCheck();
          sendResponse({ success: result.success, error: result.error });
          break;
          
        case 'updateAuthEndpoint':
          if (request.url) {
            this.licenseCheckUrl = request.url;
            await chrome.storage.local.set({ 'jeetblock_auth_endpoint': request.url });
            sendResponse({ success: true });
          } else {
            sendResponse({ success: false, error: 'No URL provided' });
          }
          break;
          
        case 'toggleExtension':
          await this.toggleExtension();
          sendResponse({ success: true });
          break;
          
        case 'updateIcon':
          await this.updateIcon(request.enabled);
          sendResponse({ success: true });
          break;
          
        default:
          sendResponse({ success: false, error: 'Unknown action' });
      }
    } catch (error) {
      console.error('JeetBlock Background: Message handling error:', error);
      sendResponse({ success: false, error: error.message });
    }
  }

  // Method to update the authentication endpoint
  async updateAuthEndpoint(newUrl) {
    this.licenseCheckUrl = newUrl;
    await chrome.storage.local.set({ 'jeetblock_auth_endpoint': newUrl });
    console.log('JeetBlock: Authentication endpoint updated to:', newUrl);
  }

  // Initialize icon based on current enabled state
  async initializeIcon() {
    try {
      const result = await chrome.storage.sync.get(['isEnabled']);
      const enabled = this.parseBoolean(result.isEnabled, false); // Default to false
      await this.updateIcon(enabled);
    } catch (error) {
      console.error('JeetBlock: Error initializing icon:', error);
    }
  }

  // Toggle extension on/off
  async toggleExtension() {
    try {
      // Get current enabled state
      const result = await chrome.storage.sync.get(['isEnabled']);
      const currentState = this.parseBoolean(result.isEnabled, false); // Default to false
      const newState = !currentState;
      
      // Update the setting
      await chrome.storage.sync.set({ isEnabled: newState });
      
      // Update icon
      await this.updateIcon(newState);
      
      // Notify all tabs to update their filtering
      const tabs = await chrome.tabs.query({});
      for (const tab of tabs) {
        try {
          await chrome.tabs.sendMessage(tab.id, {
            action: 'updateSettings',
            settings: { isEnabled: newState }
          });
        } catch (error) {
          // Tab might not have content script, ignore
        }
      }
      
      console.log('JeetBlock: Extension toggled', newState ? 'ON' : 'OFF');
    } catch (error) {
      console.error('JeetBlock: Error toggling extension:', error);
    }
  }

  // Update extension icon based on enabled state
  async updateIcon(enabled) {
    try {
      const iconPath = enabled ? {
        16: 'icons/icon16.png',
        48: 'icons/icon48.png',
        128: 'icons/icon128.png'
      } : {
        16: 'icons/icon_disabled_16.png',
        48: 'icons/icon_disabled_48.png',
        128: 'icons/icon_disabled_128.png'
      };

      await chrome.action.setIcon({ path: iconPath });
      
      // Update title based on state
      const title = enabled ? 'JeetBlock - Enabled' : 'JeetBlock - Disabled';
      await chrome.action.setTitle({ title });
      
    } catch (error) {
      console.error('JeetBlock: Error updating icon:', error);
    }
  }
}

// Initialize the background service
const jeetBlockBackground = new JeetBlockBackground();

// Load saved endpoint if available, otherwise use default
chrome.storage.local.get(['jeetblock_auth_endpoint']).then(result => {
  if (result.jeetblock_auth_endpoint) {
    jeetBlockBackground.licenseCheckUrl = result.jeetblock_auth_endpoint;
    console.log('JeetBlock: Loaded saved endpoint:', result.jeetblock_auth_endpoint);
  } else {
    console.log('JeetBlock: Using default endpoint:', jeetBlockBackground.licenseCheckUrl);
  }
});
