// Social Media Harmonizer - Content Script
console.log('JeetBlock: UPDATED VERSION LOADED - TikTok debug logs should appear');

// Security: Validate license before enabling extension
async function validateLicenseSecurity() {
  try {
    const stored = await chrome.storage.local.get([
      'jeetblock_licensed', 
      'jeetblock_server_validated', 
      'jeetblock_last_validation',
      'jeetblock_validation_count'
    ]);
    
    // If someone manually set licensed=true without server validation, disable
    if (stored.jeetblock_licensed && !stored.jeetblock_server_validated) {
      console.warn('JeetBlock: SECURITY ALERT - License set without server validation');
      await chrome.storage.local.set({ 'jeetblock_licensed': false });
      return false;
    }
    
    // If validation is too old (more than 24 hours), require re-validation
    const now = Date.now();
    const lastValidation = stored.jeetblock_last_validation || 0;
    if (stored.jeetblock_licensed && (now - lastValidation) > 24 * 60 * 60 * 1000) {
      console.warn('JeetBlock: License validation expired, requiring re-validation');
      await chrome.storage.local.set({ 'jeetblock_licensed': false });
      return false;
    }
    
    return stored.jeetblock_licensed || false;
  } catch (error) {
    console.error('JeetBlock: License validation error:', error);
    return false;
  }
}

class SocialMediaHarmonizer {
  constructor() {
    this.platform = this.getPlatform();
    this.filterMode = 'collapse'; // 'collapse' or 'remove'
    this.isEnabled = true;
    this.indianNames = [];
    this.muslimNames = [];
    this.jewishNames = [];
    this.customFilters = [];
    this.filterIndian = true;
    this.filterMuslim = true;
    this.filterJewish = true;
    this.filterLGBTQIA = true;
    this.filterUkraine = true;
    this.filterPalestine = true;
    // Platform-specific filtering
    this.filterFacebook = true;
    this.filterTwitter = true;
    this.filterInstagram = true;
    this.filterLinkedIn = true;
    this.filterTikTok = true;
    this.filterReddit = true;
    this.filterYouTube = true;
    this.indianEmojis = {
      // Indian flag
      flag: ['🇮🇳', '\u{1F1EE}\u{1F1F3}'],
      // Bangladesh flag
      bangladesh: ['🇧🇩', '\u{1F1E7}\u{1F1E9}'],
      // Sri Lanka flag
      sri_lanka: ['🇱🇰', '\u{1F1F1}\u{1F1F0}'],
      // Hindu/Indian religious symbols
      om: ['🕉️', '\u{1F549}'],
      diya: ['🪔', '\u{1FA94}'],
      // Indian cultural symbols
      lotus: ['🪷', '\u{1FAB7}'],
      bindi: [], // Red dot often used as bindi
      // Indian food
      curry: ['🍛', '\u{1F35B}'],
      // Indian animals
      cow: ['🐄', '\u{1F404}'],
      elephant: ['🐘', '\u{1F418}'],
      peacock: ['🦚', '\u{1F99A}'],
      // Indian architecture
      turban: ['👳', '\u{1F473}'],
    };

    this.muslimEmojis = {
      // Muslim country flags - Comprehensive list
      turkiye: ['🇹🇷', '\u{1F1F9}\u{1F1F7}'],
      pakistan: ['🇵🇰', '\u{1F1F5}\u{1F1F0}'],
      iran: ['🇮🇷', '\u{1F1EE}\u{1F1F7}'],
      jordan: ['🇯🇴', '\u{1F1EF}\u{1F1F4}'],
      libya: ['🇱🇾', '\u{1F1F1}\u{1F1FE}'],
      algeria: ['🇩🇿', '\u{1F1E9}\u{1F1FF}'],
      morocco: ['🇲🇦', '\u{1F1F2}\u{1F1E6}'],
      brunei: ['🇧🇳', '\u{1F1E7}\u{1F1F3}'],
      egypt: ['🇪🇬', '\u{1F1EA}\u{1F1EC}'],
      bahrain: ['🇧🇭', '\u{1F1E7}\u{1F1ED}'],
      chad: ['🇹🇩', '\u{1F1F9}\u{1F1E9}'],
      syria: ['🇸🇾', '\u{1F1F8}\u{1F1FE}'],
      iraq: ['🇮🇶', '\u{1F1EE}\u{1F1F6}'],
      saudi_arabia: ['🇸🇦', '\u{1F1F8}\u{1F1E6}'],
      lebanon: ['🇱🇧', '\u{1F1F1}\u{1F1E7}'],
      mauritania: ['🇲🇷', '\u{1F1F2}\u{1F1F7}'],
      somalia: ['🇸🇴', '\u{1F1F8}\u{1F1F4}'],
      tunisia: ['🇹🇳', '\u{1F1F9}\u{1F1F3}'],
      afghanistan: ['🇦🇫', '\u{1F1E6}\u{1F1EB}'],
      yemen: ['🇾🇪', '\u{1F1FE}\u{1F1EA}'],
      indonesia: ['🇮🇩', '\u{1F1EE}\u{1F1E9}'],
      niger: ['🇳🇪', '\u{1F1F3}\u{1F1EA}'],
      senegal: ['🇸🇳', '\u{1F1F8}\u{1F1F3}'],
      sudan: ['🇸🇩', '\u{1F1F8}\u{1F1E9}'],
      comoros: ['🇰🇲', '\u{1F1F0}\u{1F1F2}'],
      djibouti: ['🇩🇯', '\u{1F1E9}\u{1F1EF}'],
      nigeria: ['🇳🇬', '\u{1F1F3}\u{1F1EC}'],
      azerbaijan: ['🇦🇿', '\u{1F1E6}\u{1F1FF}'],
      tajikistan: ['🇹🇯', '\u{1F1F9}\u{1F1EF}'],
      uzbekistan: ['🇺🇿', '\u{1F1FA}\u{1F1FF}'],
      mali: ['🇲🇱', '\u{1F1F2}\u{1F1F1}'],
      // Islamic religious symbols
      star_crescent: ['☪️', '\u{262A}'],
      mosque: ['🕌', '\u{1F54C}'],
      minaret: ['🕌', '\u{1F54C}'], // Same as mosque
      // Islamic clothing
      hijab: ['🧕', '\u{1F9D5}'],
 // Kimono as proxy
      // Islamic food
      shawarma: ['🌯', '\u{1F32F}'],
      falafel: ['🧆', '\u{1F9C6}'],
      hummus: ['🥣', '\u{1F963}'],
      // Islamic architecture
    };

    this.jewishEmojis = {
      // Israeli flag
      israel_flag: ['🇮🇱', '\u{1F1EE}\u{1F1F1}'],
      // Jewish religious symbols
      star_of_david: ['✡️', '\u{2721}'],
      menorah: ['🕎', '\u{1F54E}'],
      torah: ['📜', '\u{1F4DC}'],
      chai: ['חי', '\u{05D7}\u{05D9}'],
    };

    this.lgbtqiaEmojis = {
      // LGBTQIA+ flags and symbols
      rainbow_flag: ['🏳️‍🌈', '\u{1F3F3}\u{FE0F}\u{200D}\u{1F308}'],
      transgender_flag: ['🏳️‍⚧️', '\u{1F3F3}\u{FE0F}\u{200D}\u{26A7}\u{FE0F}'],
      // LGBTQIA+ related emojis
      couple_men: ['👬', '\u{1F46C}'],
      couple_women: ['👭', '\u{1F46D}'],
    };

    this.ukraineEmojis = {
      // Ukrainian flag
      ukraine_flag: ['🇺🇦', '\u{1F1FA}\u{1F1E6}'],
      // Ukrainian support phrases (common in usernames)
      ukraine_support: ['🇺🇦💙💛', '\u{1F1FA}\u{1F1E6}\u{1F499}\u{1F49B}']
    };

    this.palestineEmojis = {
      // Palestinian flag
      palestine_flag: ['🇵🇸', '\u{1F1F5}\u{1F1F8}'],
      // Palestinian colors - removed all heart emojis as too generic
      // Palestinian architecture
      dome_of_rock: ['🕌', '\u{1F54C}'], // Mosque as proxy
      // Palestinian support phrases (common in usernames) - removed heart emojis
      palestine_support: ['🇵🇸', '\u{1F1F5}\u{1F1F8}'],
      free_palestine: ['🇵🇸✌️', '\u{1F1F5}\u{1F1F8}\u{270C}']
    };
    
    this.init();
  }

  // Returns list of Muslim names matched as whole words in given text
  getMatchedMuslimNames(text) {
    if (!text) return [];
    const lowerText = text.toLowerCase();
    const matched = [];
    this.muslimNames.forEach((name) => {
      const nameLower = String(name).toLowerCase();
      // Use a more robust word boundary that handles accented characters and common separators
      const regex = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${nameLower}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      if (regex.test(lowerText)) matched.push(name);
    });
    return matched;
  }

  async init() {
    console.log('JeetBlock: Content script initializing...');
    
    // Security check - validate license before proceeding
    const isLicensed = await validateLicenseSecurity();
    if (!isLicensed) {
      console.log('JeetBlock: Extension not licensed, disabling functionality');
      this.isEnabled = false;
      return;
    }
    
    // Security check - if compromised, don't initialize
    if (window.jeetBlockSecurity && window.jeetBlockSecurity.isExtensionCompromised()) {
      return;
    }
    
    await this.loadSettings();
    await this.loadIndianNames();
    await this.loadMuslimNames();
    await this.loadJewishNames();
    this.setupMutationObserver();
    this.filterExistingPosts();
    // Scoped observer for Facebook replies only
    if (this.getPlatform() === 'facebook' && this.filterFacebook) {
      console.log('JeetBlock: Setting up Facebook message observers');
      this.setupFacebookRepliesObserver();
      this.setupFacebookMessagesObserver();
      this.setupFacebookMessageThreadObserver();
      this.setupFacebookSidebarMessagesObserver();
      this.setupFacebookChatBubbleObserver();
      this.setupFacebookSuggestedFriendsObserver();
      this.setupFacebookReelsObserver();
    } else {
      console.log('JeetBlock: Facebook message observers NOT set up. Platform:', this.getPlatform(), 'FilterFacebook:', this.filterFacebook);
    }
    // Safety: force-init Facebook message observers even if platform detection failed (gated by toggle)
    if (this.filterFacebook) {
      console.log('JeetBlock: Safety init - ensuring Facebook message observers are initialized');
      try { this.setupFacebookRepliesObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookMessagesObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookMessageThreadObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookSidebarMessagesObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookChatBubbleObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookSuggestedFriendsObserver(); } catch (e) { /* ignore */ }
      try { this.setupFacebookReelsObserver(); } catch (e) { /* ignore */ }
    }
    // Scoped observer for Instagram replies
    if (this.getPlatform() === 'instagram' && this.filterInstagram) {
      this.setupInstagramRepliesObserver();
      this.setupInstagramMessagesObserver();
      this.setupInstagramMessageThreadObserver();
    }
    // Scoped observer for Twitter messages
    if (this.getPlatform() === 'twitter' && this.filterTwitter) {
      console.log('JeetBlock: ⚡ Setting up Twitter message observers');
      this.setupTwitterMessagesObserver();
      this.setupTwitterMainMessagingObserver();
      this.setupTwitterConversationDetailsObserver();
      console.log('JeetBlock: ✅ All Twitter message observers set up complete');
    } else {
      console.log('JeetBlock: Twitter message observers NOT set up. Platform:', this.getPlatform(), 'FilterTwitter:', this.filterTwitter);
    }
    // Safety: force-init Twitter DM observers even if platform detection failed (gated by toggle)
    if (this.filterTwitter) {
      console.log('JeetBlock: Safety init - ensuring Twitter message observers are initialized');
      try { this.setupTwitterMessagesObserver(); } catch (e) { /* ignore */ }
      try { this.setupTwitterMainMessagingObserver(); } catch (e) { /* ignore */ }
      try { this.setupTwitterConversationDetailsObserver(); } catch (e) { /* ignore */ }
    }
    // Scoped observer for TikTok replies
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokRepliesObserver();
    }
    // Scoped observer for TikTok search results
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokSearchObserver();
    }
    // Scoped observer for TikTok comment items (right panel and below-video)
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokCommentItemsObserver();
    }
    // Scoped observer for TikTok suggested users (Search page)
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokSuggestedUsersObserver();
    }
    // Scoped observer for TikTok full-screen video/profile page
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokFullscreenObserver();
    }
    // Scoped observer for TikTok messages page
    if (this.getPlatform() === 'tiktok' && this.filterTikTok) {
      this.setupTikTokMessagesObserver();
      this.setupTikTokChatContentObserver();
    } else {
      console.log('JeetBlock: TikTok message observers NOT set up. Platform:', this.getPlatform(), 'FilterTikTok:', this.filterTikTok);
    }
    // Safety: force-init TikTok message observers even if platform detection failed (gated by toggle)
    console.log('JeetBlock: Checking TikTok safety fallback - filterTikTok:', this.filterTikTok);
    if (this.filterTikTok) {
      console.log('JeetBlock: Safety init - ensuring TikTok message observers are initialized');
      try { this.setupTikTokMessagesObserver(); } catch (e) { console.log('JeetBlock: Error setting up TikTok messages observer:', e); }
      try { this.setupTikTokChatContentObserver(); } catch (e) { console.log('JeetBlock: Error setting up TikTok chat content observer:', e); }
    } else {
      console.log('JeetBlock: TikTok safety fallback SKIPPED - filterTikTok is false');
    }
    // Scoped observer for Reddit comments/replies
    if (this.getPlatform() === 'reddit' && this.filterReddit) {
      this.setupRedditRepliesObserver();
      this.setupRedditMainChatObserver();
      this.setupRedditMessagesListObserver();
      this.setupRedditRoomObserver();
      this.setupRedditConversationHeaderObserver();
    }
    // Scoped observer for YouTube replies
    if (this.getPlatform() === 'youtube' && this.filterYouTube) {
      this.setupYouTubeRepliesObserver();
    }
    // Scoped observer for YouTube videos
    if (this.getPlatform() === 'youtube' && this.filterYouTube) {
      this.setupYouTubeVideosObserver();
    }
    // Scoped observer for YouTube search results
    if (this.getPlatform() === 'youtube' && this.filterYouTube) {
      this.setupYouTubeSearchObserver();
    }
    // Scoped observer for YouTube sidebar
    if (this.getPlatform() === 'youtube' && this.filterYouTube) {
      this.setupYouTubeSidebarObserver();
    }
    // Scoped observer for LinkedIn replies
    if (this.getPlatform() === 'linkedin' && this.filterLinkedIn) {
      this.setupLinkedInRepliesObserver();
      this.setupLinkedInSidebarMessagesObserver();
      this.setupLinkedInChatDialogObserver();
      this.setupLinkedInMainMessagesObserver();
      this.setupLinkedInMainMessagingObserver();
      this.setupLinkedInConversationHeaderObserver();
    }
    // Scoped observer for LinkedIn main posts
    if (this.getPlatform() === 'linkedin' && this.filterLinkedIn) {
      this.setupLinkedInPostsObserver();
    }
    // Scoped observer for LinkedIn employee listings
    if (this.getPlatform() === 'linkedin' && this.filterLinkedIn) {
      this.setupLinkedInEmployeesObserver();
    }
    // Scoped observer for LinkedIn reshared posts
    if (this.getPlatform() === 'linkedin' && this.filterLinkedIn) {
      this.setupLinkedInResharesObserver();
    }
    // Scoped observer for LinkedIn search results
    if (this.getPlatform() === 'linkedin' && this.filterLinkedIn) {
      this.setupLinkedInSearchResultsObserver();
    }
    
    console.log('JeetBlock: Content script initialized successfully');
  }

  // Universal Instagram Search Sidebar Exemption
  isInstagramSearchSidebar(element) {
    if (!window.location.hostname.includes('instagram.com')) return false;
    
    // Check if element is the search sidebar container or within it
    if (element.classList && element.classList.contains('x7goit8') && 
        element.classList.contains('x10l6tqk') && element.classList.contains('x1ja2u2z')) {
      return true;
    }
    
    // Check if element is within the search sidebar
    const searchSidebar = element.closest('div.x7goit8.x10l6tqk.x1ja2u2z');
    if (searchSidebar) {
      return true;
    }
    
    return false;
  }

  // ========================= Instagram Replies (Scoped) =========================
  getInstagramReplyUserSelector() {
    // Profile anchor that wraps the avatar image with alt "<user>'s profile picture"
    return 'a[role="link"] img[alt*="profile picture"]';
  }

  setupInstagramRepliesObserver() {
    try {
      const userSel = this.getInstagramReplyUserSelector();
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            const direct = node.matches && node.matches(userSel) ? [node] : [];
            const inner = node.querySelectorAll ? node.querySelectorAll(userSel) : [];
            if (direct.length) pending.push(...direct);
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (!pending.length) return;
        requestAnimationFrame(() => {
          pending.forEach((imgEl) => {
            const anchor = imgEl.closest('a[role="link"]');
            if (!anchor) return;
            this.filterInstagramReplyFromAnchor(anchor);
          });
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(userSel).forEach((imgEl) => {
        const anchor = imgEl.closest('a[role="link"]');
        if (anchor) this.filterInstagramReplyFromAnchor(anchor);
      });
      console.log('JeetBlock: Instagram replies observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Instagram replies observer', e);
    }
  }

  filterInstagramReplyFromAnchor(profileAnchor) {
    if (!this.isEnabled || !this.filterInstagram) return;
    
    // Universal Instagram Search Sidebar Exemption
    if (this.isInstagramSearchSidebar(profileAnchor)) {
      console.log('JeetBlock: Instagram search sidebar detected in replies - exempting from filtering');
      return;
    }
    try {
      const href = profileAnchor.getAttribute('href') || '';
      const m = href.match(/^\/([^\/]+)\/?$/);
      const username = m ? m[1] : '';
      let container = profileAnchor;
      for (let i = 0; i < 8 && container; i++) {
        if (container.querySelector && container.querySelector('a[href*="/p/"][href*="/c/"] time')) break;
        container = container.parentElement;
      }
      if (!container) container = profileAnchor.parentElement;
      if (!container || container.dataset.harmonizerProcessed === 'true') return;
      container.dataset.harmonizerProcessed = 'true';
      // Mark as Instagram reply for custom UI treatment
      container.dataset.igReply = 'true';

      let contentText = '';
      const contentSpan = container.querySelector('span[dir="auto"]');
      if (contentSpan) contentText = this.getTextWithEmojis(contentSpan);

      const combinedLower = ((username || '') + ' ' + (contentText || '')).toLowerCase();
      let shouldFilter = false; const reasons = [];

      if (this.filterIndian && (
        this.containsDevanagari(combinedLower) ||
        this.containsBengali(combinedLower) ||
        this.containsSinhala(combinedLower) ||
        this.containsGurmukhi(combinedLower) ||
        this.containsIndianName(combinedLower) ||
        this.containsIndianEmoji(username) || this.containsIndianEmoji(contentText)
      )) { shouldFilter = true; reasons.push('Indian match'); }

      if (!shouldFilter && this.filterMuslim) {
        const muslimReasons = [];
        if (this.containsArabic(combinedLower)) muslimReasons.push('Arabic script');
        if (this.containsPersian(combinedLower)) muslimReasons.push('Persian script');
        if (this.containsUrdu(combinedLower)) muslimReasons.push('Urdu script');
        const matched = this.getMatchedMuslimNames(combinedLower);
        if (matched.length) muslimReasons.push(`Muslim name [${matched.join(', ')}]`);
        if (this.containsMuslimEmoji(username) || this.containsMuslimEmoji(contentText)) muslimReasons.push('Muslim emoji');
        if (muslimReasons.length) { shouldFilter = true; reasons.push(...muslimReasons); }
      }

      if (!shouldFilter && this.filterJewish) {
        if (this.containsHebrew(combinedLower) || this.containsJewishName(combinedLower) ||
            this.containsJewishEmoji(username) || this.containsJewishEmoji(contentText)) {
          shouldFilter = true; reasons.push('Jewish match');
        }
      }

      if (!shouldFilter) { delete container.dataset.harmonizerProcessed; return; }
      console.log(`JeetBlock: IG-Comment: Filtering reply by ${username || '[unknown]'} (reasons: ${reasons.join(', ')})`);
      container.dataset.instagram = 'true';
      this.applyFilter(container);
    } catch (e) {
      console.warn('JeetBlock: IG-Comment error', e);
    }
  }

  // ========================= Instagram Messages (Scoped) =========================
  getInstagramMessageSelector() {
    // Button/row that represents a DM thread entry in IG inbox
    return 'div[role="button"][tabindex="0"] span.xlyipyv.xuxw1ft';
  }

  setupInstagramMessagesObserver() {
    try {
      const selector = this.getInstagramMessageSelector();
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (!pending.length) return;
        requestAnimationFrame(() => pending.forEach((n) => this.filterInstagramMessageFromNameSpan(n)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((n) => this.filterInstagramMessageFromNameSpan(n));
      console.log('JeetBlock: Instagram messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Instagram messages observer', e);
    }
  }

  filterInstagramMessageFromNameSpan(nameSpan) {
    if (!this.isEnabled || !this.filterInstagram) return;
    
    // Universal Instagram Search Sidebar Exemption
    if (this.isInstagramSearchSidebar(nameSpan)) {
      console.log('JeetBlock: Instagram search sidebar detected in messages - exempting from filtering');
      return;
    }
    try {
      const container = nameSpan.closest('div[role="button"], [role="link"]') || nameSpan.parentElement;
      if (!container || container.dataset.harmonizerProcessed === 'true') return;
      container.dataset.harmonizerProcessed = 'true';
      container.dataset.igMessage = 'true';

      const username = this.getTextWithEmojis(nameSpan);
      if (!username) return;

      // Pull nearby preview text
      let contentText = '';
      const siblingPreview = container.querySelector('span[dir="auto"]');
      if (siblingPreview) contentText = this.getTextWithEmojis(siblingPreview);

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
      if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
      if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }

      if (this.containsDevanagari(fullTextLower)) { shouldFilter = true; reasons.push('Devanagari script'); }
      if (this.containsBengali(fullTextLower)) { shouldFilter = true; reasons.push('Bengali script'); }
      if (this.containsSinhala(fullTextLower)) { shouldFilter = true; reasons.push('Sinhala script'); }
      if (this.containsGurmukhi(fullTextLower)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
      if (this.containsIndianEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Indian emoji'); }

      if (!shouldFilter) {
        delete container.dataset.harmonizerProcessed;
        return;
      }

      this.applyFilter(container, 'instagram-message');
      console.log(`JeetBlock: IG-Message: Filtering message from ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: IG-Message error', e);
    }
  }

  // When a filtered IG DM is opened, collapse the thread grid as well
  setupInstagramMessageThreadObserver() {
    try {
      const selector = 'div[role="grid"][aria-label^="Messages in conversation with"], [data-pagelet="IGDOpenMessageList"] div[role="grid"]';
      const observer = new MutationObserver((mutations) => {
        const grids = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) grids.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => grids.add(el));
          });
        }
        if (!grids.size) return;
        requestAnimationFrame(() => grids.forEach((grid) => this.filterInstagramMessageThread(grid)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((grid) => this.filterInstagramMessageThread(grid));
      console.log('JeetBlock: Instagram message thread observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Instagram message thread observer', e);
    }
  }

  filterInstagramMessageThread(gridEl) {
    if (!this.isEnabled || !this.filterInstagram) return;
    
    // Universal Instagram Search Sidebar Exemption
    if (this.isInstagramSearchSidebar(gridEl)) {
      console.log('JeetBlock: Instagram search sidebar detected in message thread - exempting from filtering');
      return;
    }
    try {
      if (!gridEl || gridEl.dataset.harmonizerProcessed === 'true') return;

      // Extract the header name; IG uses h2 with span or aria-label on grid
      let username = '';
      const headerSpan = document.querySelector('h2 span.xlyipyv.xuxw1ft span[title]');
      if (headerSpan && headerSpan.getAttribute('title')) username = headerSpan.getAttribute('title');
      if (!username && gridEl.getAttribute('aria-label')) {
        const aria = gridEl.getAttribute('aria-label');
        const m = aria.match(/Messages in conversation with\s+(.+)$/i);
        if (m) username = m[1].trim();
      }
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      if (this.containsJewishName(usernameLower)) shouldFilter = true;
      if (this.containsMuslimName(usernameLower)) shouldFilter = true;
      if (this.containsIndianName(usernameLower)) shouldFilter = true;

      if (!shouldFilter) return;

      gridEl.dataset.harmonizerProcessed = 'true';
      gridEl.dataset.igMessageThread = 'true';
      this.applyFilter(gridEl, 'instagram-message-thread');
      console.log(`JeetBlock: IG-Thread: Collapsed thread for ${username}`);
    } catch (e) {
      console.warn('JeetBlock: IG-Thread error', e);
    }
  }

  // ========================= TikTok Replies (Scoped) =========================
  setupTikTokRepliesObserver() {
    try {
      const selector = '[data-e2e^="comment-username"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (!pending.length) return;
        requestAnimationFrame(() => {
          pending.forEach((userNode) => this.filterTikTokReplyFromUserNode(userNode));
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Also observe the right-rail wrapper if present; retry a few times if not yet mounted
      const tryAttachRightRail = () => {
        const rightRail = document.querySelector('div[class*="--DivProfileWrapper"]');
        if (rightRail) {
          console.log('JeetBlock: Observing right-rail wrapper for reply usernames');
          try { observer.observe(rightRail, { childList: true, subtree: true }); } catch (_) {}
          return true;
        }
        return false;
      };
      if (!tryAttachRightRail()) {
        let attempts = 0;
        const intId = setInterval(() => {
          if (tryAttachRightRail() || ++attempts >= 10) clearInterval(intId);
        }, 300);
      }
      document.querySelectorAll(selector).forEach((n) => this.filterTikTokReplyFromUserNode(n));
      console.log('JeetBlock: TikTok replies observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize TikTok replies observer', e);
    }
  }

  // TikTok comment items observer (covers right-side and under-video mounts)
  setupTikTokCommentItemsObserver() {
    try {
      console.log('JeetBlock: Setting up TikTok comment items observer');
      const selector = 'div[class*="--DivCommentItemContainer"], div[class*="--DivCommentContentContainer"], div[class*="--DivCommentItemWrapper"], div[class*="--DivCommentObjectWrapper"], div[class*="--DivCommentContentWrapper"], div[data-e2e^="comments-level"], div[data-e2e^="comment-item"], li[data-e2e^="comment-item"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (!pending.length) return;
        requestAnimationFrame(() => {
          pending.forEach((el) => this.filterTikTokCommentItem(el));
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Also observe the right-rail wrapper, retry for a bit if it isn't mounted yet
      const tryAttachRightRail = () => {
        const rightRail = document.querySelector('div[class*="--DivProfileWrapper"]');
        if (rightRail) {
          console.log('JeetBlock: Observing right-rail profile wrapper for comments');
          try { observer.observe(rightRail, { childList: true, subtree: true }); } catch (_) {}
          return true;
        }
        return false;
      };
      if (!tryAttachRightRail()) {
        let attempts = 0;
        const intId = setInterval(() => {
          if (tryAttachRightRail() || ++attempts >= 10) clearInterval(intId);
        }, 300);
      }
      document.querySelectorAll(selector).forEach((el) => this.filterTikTokCommentItem(el));
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize TikTok comment items observer', e);
    }
  }

  filterTikTokCommentItem(container) {
    if (!this.isEnabled || !this.filterTikTok || !container) return;
    try {
      if (container.dataset.harmonizerProcessed === 'true') return;
      // Username
      let username = '';
      const usernameSpan = container.querySelector('[data-e2e^="comment-username"], span[class*="--SpanUserNameText"], a[href^="/@"] [class*="--StyledTUXText"], a[href^="/@"] p');
      if (usernameSpan) username = this.getTextWithEmojis(usernameSpan);
      if (!username) {
        const link = container.querySelector('a[href^="/@"]');
        const href = link ? (link.getAttribute('href') || '') : '';
        const m = href.match(/^\/@([^/?#]+)/);
        if (m) username = m[1];
      }
      // Comment text
      let contentText = '';
      const commentP = container.querySelector('p[data-e2e^="comment-level"], p[role="text"]');
      if (commentP) contentText = this.getTextWithEmojis(commentP);
      if (!contentText) {
        const span = container.querySelector('span[dir], span[class*="StyledTUXText"]');
        if (span) contentText = this.getTextWithEmojis(span);
      }

      const usernameLower = (username || '').toLowerCase();
      const combinedLower = (usernameLower + ' ' + (contentText || '').toLowerCase());

      let shouldFilter = false; const reasons = [];

      // Indian
      if (this.filterIndian) {
        const indianReasons = [];
        if (this.containsIndianName(usernameLower)) indianReasons.push('Indian name');
        if (this.containsDevanagari(combinedLower) || this.containsBengali(combinedLower) || this.containsSinhala(combinedLower) || this.containsGurmukhi(combinedLower)) indianReasons.push('Indian script');
        if (this.containsIndianEmoji(username) || this.containsIndianEmoji(contentText)) indianReasons.push('Indian emoji');
        if (indianReasons.length) { shouldFilter = true; reasons.push(...indianReasons); }
      }

      // Muslim
      if (!shouldFilter && this.filterMuslim) {
        const muslimReasons = [];
        if (this.containsArabic(combinedLower)) muslimReasons.push('Arabic script');
        if (this.containsPersian(combinedLower)) muslimReasons.push('Persian script');
        if (this.containsUrdu(combinedLower)) muslimReasons.push('Urdu script');
        const matched = this.getMatchedMuslimNames(usernameLower);
        if (matched.length) muslimReasons.push(`Muslim name [${matched.join(', ')}]`);
        if (this.containsMuslimEmoji(username) || this.containsMuslimEmoji(contentText)) muslimReasons.push('Muslim emoji');
        if (muslimReasons.length) { shouldFilter = true; reasons.push(...muslimReasons); }
      }

      // Jewish
      if (!shouldFilter && this.filterJewish) {
        const jewishReasons = [];
        if (this.containsJewishName(usernameLower)) jewishReasons.push('Jewish name');
        if (this.containsHebrew(combinedLower)) jewishReasons.push('Hebrew script');
        if (this.containsJewishEmoji(username) || this.containsJewishEmoji(contentText)) jewishReasons.push('Jewish emoji');
        if (jewishReasons.length) { shouldFilter = true; reasons.push(...jewishReasons); }
      }

      // LGBTQIA / Ukraine / Palestine
      if (!shouldFilter && this.filterLGBTQIA && this.containsLGBTQIAEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('LGBTQIA+ emoji');
      }
      if (!shouldFilter && this.filterUkraine && this.containsUkraineEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('Ukraine flag');
      }
      if (!shouldFilter && this.filterPalestine && this.containsPalestineEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('Palestine flag');
      }

      if (!shouldFilter) return;
      container.dataset.harmonizerProcessed = 'true';
      container.dataset.tiktokReply = 'true';
      console.log(`JeetBlock: TT-CommentItem: Filtering by ${username || '[unknown]'} (reasons: ${reasons.join(', ')})`);
      this.applyFilter(container);
    } catch (e) {
      console.warn('JeetBlock: TT-CommentItem error', e);
    }
  }

  filterTikTokReplyFromUserNode(userNode) {
    if (!this.isEnabled || !this.filterTikTok) return;
    try {
      // Extract username: try direct text first (span has the name), then derive from closest link
      let username = '';
      const directText = (userNode.textContent || '').trim();
      if (directText) username = directText;
      if (!username) {
        const parentLink = userNode.closest('a[href^="/@"]');
        if (parentLink) {
          const href = parentLink.getAttribute('href') || '';
        const m = href.match(/^\/@([^/?#]+)/);
        if (m) username = m[1];
        }
      }

      // Find a stable container for the whole comment item
      let container = userNode.closest('div[class*="--DivCommentItemWrapper"]') ||
                      userNode.closest('div[class*="--DivCommentObjectWrapper"]') ||
                      userNode.closest('div[class*="--DivCommentContentContainer"]') ||
                      userNode.closest('div[class*="--DivCommentItemContainer"]') ||
                      userNode;
      if (!container || container.dataset.harmonizerProcessed === 'true') return;
      container.dataset.harmonizerProcessed = 'true';
      container.dataset.tiktokReply = 'true';

      // Extract comment content (scripts/emojis may use this)
      let contentText = '';
      // Prefer the comment paragraph
      const commentP = container.querySelector('p[data-e2e^="comment-level"]');
      if (commentP && commentP.textContent) {
        contentText = commentP.textContent.trim();
      } else {
        // Fallback to generic spans (dir may be empty, not only "auto")
        const textSpan = container.querySelector('span[class*="StyledTUXText"], span[dir]');
      if (textSpan) contentText = (textSpan.textContent || '').trim();
      }

      const usernameLower = (username || '').toLowerCase();
      const combinedLower = (usernameLower + ' ' + (contentText || '').toLowerCase());
      let shouldFilter = false; const reasons = [];

      // Indian (name only on username; scripts/emojis can use content)
      if (this.filterIndian) {
        const indianReasons = [];
        if (this.containsIndianName(usernameLower)) indianReasons.push('Indian name');
        if (this.containsDevanagari(combinedLower) || this.containsBengali(combinedLower) || this.containsSinhala(combinedLower) || this.containsGurmukhi(combinedLower)) indianReasons.push('Indian script');
        if (this.containsIndianEmoji(username) || this.containsIndianEmoji(contentText)) indianReasons.push('Indian emoji');
        if (indianReasons.length) { shouldFilter = true; reasons.push(...indianReasons); }
      }

      // Muslim (names on username only)
      if (!shouldFilter && this.filterMuslim) {
        const muslimReasons = [];
        if (this.containsArabic(combinedLower)) muslimReasons.push('Arabic script');
        if (this.containsPersian(combinedLower)) muslimReasons.push('Persian script');
        if (this.containsUrdu(combinedLower)) muslimReasons.push('Urdu script');
        const matched = this.getMatchedMuslimNames(usernameLower);
        if (matched.length) muslimReasons.push(`Muslim name [${matched.join(', ')}]`);
        if (this.containsMuslimEmoji(username) || this.containsMuslimEmoji(contentText)) muslimReasons.push('Muslim emoji');
        if (muslimReasons.length) { shouldFilter = true; reasons.push(...muslimReasons); }
      }

      // Jewish (names on username only)
      if (!shouldFilter && this.filterJewish) {
        const jewishReasons = [];
        if (this.containsJewishName(usernameLower)) jewishReasons.push('Jewish name');
        if (this.containsHebrew(combinedLower)) jewishReasons.push('Hebrew script');
        if (this.containsJewishEmoji(username) || this.containsJewishEmoji(contentText)) jewishReasons.push('Jewish emoji');
        if (jewishReasons.length) { shouldFilter = true; reasons.push(...jewishReasons); }
      }

      // LGBTQIA / Ukraine / Palestine flags in username or comment content
      if (!shouldFilter && this.filterLGBTQIA && this.containsLGBTQIAEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('LGBTQIA+ emoji');
      }
      if (!shouldFilter && this.filterUkraine && this.containsUkraineEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('Ukraine flag');
      }
      if (!shouldFilter && this.filterPalestine && this.containsPalestineEmoji((username || '') + ' ' + contentText)) {
        shouldFilter = true; reasons.push('Palestine flag');
      }

      if (!shouldFilter) { delete container.dataset.harmonizerProcessed; return; }
      console.log(`JeetBlock: TT-Comment: Filtering reply by ${username || '[unknown]'} (reasons: ${reasons.join(', ')})`);
      container.dataset.tiktok = 'true';
      this.applyFilter(container);
    } catch (e) {
      console.warn('JeetBlock: TT-Comment error', e);
    }
  }

  // ========================= Reddit Replies (Scoped) =========================
  setupRedditRepliesObserver() {
    try {
      const selector = 'shreddit-comment[author]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (!pending.length) return;
        requestAnimationFrame(() => { pending.forEach((el) => this.filterRedditComment(el)); });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((el) => this.filterRedditComment(el));
      console.log('JeetBlock: Reddit replies observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Reddit replies observer', e);
    }
  }

  filterRedditComment(commentEl) {
    if (!this.isEnabled || !this.filterReddit) return;
    try {
      if (!commentEl || commentEl.dataset.harmonizerProcessed === 'true') return;
      const author = (commentEl.getAttribute('author') || '').trim();
      // Depth info is available; we don't cascade hide children unless their author matches too
      const depth = commentEl.getAttribute('depth');

      // Build a minimal content string for script/emoji checks without causing cascade filtering
      const contentSlot = commentEl.querySelector('[slot="comment"]');
      const contentText = contentSlot ? (contentSlot.textContent || '').trim() : '';

      const usernameLower = author.toLowerCase();
      const combinedLower = (usernameLower + ' ' + contentText.toLowerCase());
      let shouldFilter = false; const reasons = [];

      // Indian
      if (this.filterIndian) {
        const indianReasons = [];
        if (this.containsIndianName(usernameLower)) indianReasons.push('Indian name');
        if (this.containsDevanagari(combinedLower) || this.containsBengali(combinedLower) || this.containsSinhala(combinedLower) || this.containsGurmukhi(combinedLower)) indianReasons.push('Indian script');
        if (this.containsIndianEmoji(author) || this.containsIndianEmoji(contentText)) indianReasons.push('Indian emoji');
        if (indianReasons.length) { shouldFilter = true; reasons.push(...indianReasons); }
      }

      // Muslim
      if (!shouldFilter && this.filterMuslim) {
        const muslimReasons = [];
        if (this.containsArabic(combinedLower)) muslimReasons.push('Arabic script');
        if (this.containsPersian(combinedLower)) muslimReasons.push('Persian script');
        if (this.containsUrdu(combinedLower)) muslimReasons.push('Urdu script');
        const matched = this.getMatchedMuslimNames(usernameLower);
        if (matched.length) muslimReasons.push(`Muslim name [${matched.join(', ')}]`);
        if (this.containsMuslimEmoji(author) || this.containsMuslimEmoji(contentText)) muslimReasons.push('Muslim emoji');
        if (muslimReasons.length) { shouldFilter = true; reasons.push(...muslimReasons); }
      }

      // Jewish
      if (!shouldFilter && this.filterJewish) {
        const jewishReasons = [];
        if (this.containsJewishName(usernameLower)) jewishReasons.push('Jewish name');
        if (this.containsHebrew(combinedLower)) jewishReasons.push('Hebrew script');
        if (this.containsJewishEmoji(author) || this.containsJewishEmoji(contentText)) jewishReasons.push('Jewish emoji');
        if (jewishReasons.length) { shouldFilter = true; reasons.push(...jewishReasons); }
      }

      if (!shouldFilter) return;
      console.log(`JeetBlock: RD-Comment: Filtering comment by ${author || '[unknown]'} (depth: ${depth ?? 'n/a'}) (reasons: ${reasons.join(', ')})`);
      commentEl.dataset.harmonizerProcessed = 'true';
      commentEl.dataset.reddit = 'true';
      this.applyFilter(commentEl);
    } catch (e) {
      console.warn('JeetBlock: RD-Comment error', e);
    }
  }

  // ========================= Reddit Main Chat Window =========================
  setupRedditMainChatObserver() {
    try {
      const selector = 'rs-timeline';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterRedditMainChat(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterRedditMainChat(item));
      console.log('JeetBlock: Reddit main chat observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Reddit main chat observer', e);
    }
  }

  filterRedditMainChat(timelineEl) {
    if (!this.isEnabled || !this.filterReddit) return;
    try {
      if (!timelineEl || timelineEl.dataset.harmonizerProcessed === 'true') return;

      // Skip if this is part of the message composer
      if (timelineEl.closest('rs-message-composer')) return;

      // Extract username from the conversation header (shadow-friendly)
      let username = '';
      const getDeepText = (el) => {
        let t = '';
        try { t += (el.textContent || '').trim(); if (el.shadowRoot) t += ' ' + (el.shadowRoot.textContent || '').trim(); } catch (_) {}
        return t.trim();
      };

      // Prefer explicit aria-label with direct chat wording
      const ariaHeader = document.querySelector('[aria-label*="Direct chat with"], [aria-label*="Current chat"]');
      if (ariaHeader) {
        const label = ariaHeader.getAttribute('aria-label') || '';
        let m = label.match(/Direct chat with\s+([^,]+)/i) || label.match(/Current chat,\s*Direct chat with\s*([^,]+)/i);
        if (m && m[1]) username = m[1].trim();
      }
      // Fallback: text-bearing title elements in header area
      if (!username) {
        const headerTitle = document.querySelector('header .text-14, header [title], header [class*="title"]') || document.querySelector('rs-room header');
        if (headerTitle) username = getDeepText(headerTitle);
      }

      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories
      if (this.filterMuslim && this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.filterJewish && this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.filterIndian && this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ content');
      }
      if (this.filterUkraine && this.containsUkraineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine content');
      }
      if (this.filterPalestine && this.containsPalestineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine content');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      timelineEl.dataset.harmonizerProcessed = 'true';
      timelineEl.dataset.redditMainChat = 'true';
      this.applyFilter(timelineEl, 'reddit-main-chat');
      console.log(`JeetBlock: Reddit Main Chat filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Reddit Main Chat filter error', e);
    }
  }

  // ========================= Reddit Messages List =========================
  setupRedditMessagesListObserver() {
    try {
      const selector = 'rs-rooms-nav-room';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterRedditMessagesList(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterRedditMessagesList(item));
      console.log('JeetBlock: Reddit messages list observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Reddit messages list observer', e);
    }
  }

  filterRedditMessagesList(roomElement) {
    if (!this.isEnabled || !this.filterReddit) return;
    try {
      if (!roomElement || roomElement.dataset.harmonizerProcessed === 'true') return;

      console.log('JeetBlock: Reddit Messages List - Processing room element');

      // Extract username from various possible sources
      let username = '';

      // Helper: deep text, traversing open shadow roots
      const getDeepText = (el) => {
        let text = '';
        try {
          text += (el.textContent || '').trim();
          if (el.shadowRoot) {
            text += ' ' + (el.shadowRoot.textContent || '').trim();
          }
        } catch (_) {}
        return text.trim();
      };

      // Try to get username from deep text content first (covers shadow DOM)
      const deepText = getDeepText(roomElement);
      if (deepText) username = deepText;
      
      // Try to look for username in child elements
      if (!username) {
        const usernameEl = roomElement.querySelector('.text-12, .text-14, [class*="username"], [class*="name"]');
        if (usernameEl) {
          username = usernameEl.textContent.trim();
        }
      }
      
      // Try to get username from room attribute as fallback
      if (!username) {
        const roomAttr = roomElement.getAttribute('room');
        if (roomAttr) {
          // Room format: "!8-zjor-G9Gdg04J1f3enHpHt8Lnp2Sei15NKbaxvDEY:reddit.com"
          // The username might be encoded in the ID part
          const match = roomAttr.match(/!([^:]+):reddit\.com$/);
          if (match) {
            username = match[1];
          }
        }
      }
      
      // Try to get username from any data attributes
      if (!username) {
        const dataUser = roomElement.getAttribute('data-user') || 
                        roomElement.getAttribute('data-username') ||
                        roomElement.getAttribute('data-name');
        if (dataUser) {
          username = dataUser;
        }
      }

      if (!username) {
        return;
      }
      
      console.log('JeetBlock: Reddit Messages List - Username:', username);

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories
      if (this.filterMuslim && this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.filterJewish && this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.filterIndian && this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ content');
      }
      if (this.filterUkraine && this.containsUkraineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine content');
      }
      if (this.filterPalestine && this.containsPalestineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine content');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      roomElement.dataset.harmonizerProcessed = 'true';
      roomElement.dataset.redditMessagesList = 'true';
      this.applyFilter(roomElement, 'reddit-messages-list');
      console.log(`JeetBlock: Reddit Messages List filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Reddit Messages List filter error', e);
    }
  }

  // ========================= Reddit Room/Conversation View =========================
  setupRedditRoomObserver() {
    try {
      const selector = 'rs-room';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterRedditRoom(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterRedditRoom(item));
      console.log('JeetBlock: Reddit room observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Reddit room observer', e);
    }
  }

  filterRedditRoom(roomElement) {
    if (!this.isEnabled || !this.filterReddit) return;
    try {
      if (!roomElement || roomElement.dataset.harmonizerProcessed === 'true') return;

      // Extract username from the room attribute
      let username = '';
      
      const roomAttr = roomElement.getAttribute('room');
      if (roomAttr) {
        // Room format: "!8-zjor-G9Gdg04J1f3enHpHt8Lnp2Sei15NKbaxvDEY:reddit.com"
        // The username might be encoded in the ID part, but we need to check if this matches
        // a known filtered user by looking at the URL or other context
        
        // For now, let's check if we can find the username in the page context
        // Look for any elements that might contain the username
        const pageContent = document.body.textContent || '';
        
        // Check if any of our filtered names appear in the page content
        const jewishNames = this.jewishNames || [];
        const muslimNames = this.muslimNames || [];
        const indianNames = this.indianNames || [];
        
        for (const name of jewishNames) {
          if (pageContent.toLowerCase().includes(name.toLowerCase())) {
            username = name;
            break;
          }
        }
        
        if (!username) {
          for (const name of muslimNames) {
            if (pageContent.toLowerCase().includes(name.toLowerCase())) {
              username = name;
              break;
            }
          }
        }
        
        if (!username) {
          for (const name of indianNames) {
            if (pageContent.toLowerCase().includes(name.toLowerCase())) {
              username = name;
              break;
            }
          }
        }
      }

      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories
      if (this.filterMuslim && this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.filterJewish && this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.filterIndian && this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ content');
      }
      if (this.filterUkraine && this.containsUkraineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine content');
      }
      if (this.filterPalestine && this.containsPalestineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine content');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      roomElement.dataset.harmonizerProcessed = 'true';
      roomElement.dataset.redditRoom = 'true';
      this.applyFilter(roomElement, 'reddit-room');
      console.log(`JeetBlock: Reddit Room filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Reddit Room filter error', e);
    }
  }

  // ========================= Reddit Conversation Header =========================
  setupRedditConversationHeaderObserver() {
    try {
      const selector = 'header[aria-label*="Current chat"], header .text-14[aria-label*="Current chat"], header .text-14';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterRedditConversationFromHeader(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterRedditConversationFromHeader(item));
      console.log('JeetBlock: Reddit conversation header observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Reddit conversation header observer', e);
    }
  }

  filterRedditConversationFromHeader(headerEl) {
    if (!this.isEnabled || !this.filterReddit) return;
    try {
      if (!headerEl || headerEl.dataset.harmonizerProcessed === 'true') return;

      // Extract username from aria-label or text content
      let username = '';
      const aria = headerEl.getAttribute && headerEl.getAttribute('aria-label');
      if (aria) {
        const m = aria.match(/Current chat,\s*Direct chat with\s*(.+)/i);
        if (m) username = m[1].trim();
      }
      if (!username) {
        const titleDiv = headerEl.querySelector('[aria-label*="Current chat"], .text-14[title], .text-14');
        if (titleDiv) username = (titleDiv.textContent || '').trim();
      }
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories
      if (this.filterMuslim && this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.filterJewish && this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.filterIndian && this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ content');
      }
      if (this.filterUkraine && this.containsUkraineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine content');
      }
      if (this.filterPalestine && this.containsPalestineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine content');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      // Filter the main timeline and selected room in sidebar
      const timeline = document.querySelector('rs-timeline');
      if (timeline && timeline.dataset.harmonizerProcessed !== 'true') {
        timeline.dataset.harmonizerProcessed = 'true';
        timeline.dataset.redditConversationTimeline = 'true';
        this.applyFilter(timeline, 'reddit-conversation-timeline');
      }

      const selectedRoom = document.querySelector('rs-rooms-nav-room[selected]');
      if (selectedRoom && selectedRoom.dataset.harmonizerProcessed !== 'true') {
        selectedRoom.dataset.harmonizerProcessed = 'true';
        selectedRoom.dataset.redditMessagesList = 'true';
        this.applyFilter(selectedRoom, 'reddit-messages-list');
      }

      headerEl.dataset.harmonizerProcessed = 'true';
      headerEl.dataset.redditConversationHeader = 'true';
      console.log(`JeetBlock: Reddit conversation filtered from header: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Reddit conversation header filter error', e);
    }
  }

  // Strict selector to target ONLY Facebook comment/reply articles
  getFacebookReplySelector() {
    return 'div[role="article"][aria-label^="Comment by"]';
  }

  // Observe only Facebook replies; avoid broad post selectors to prevent over-filtering
  setupFacebookRepliesObserver() {
    try {
      const selector = this.getFacebookReplySelector();
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const mutation of mutations) {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const innerMatches = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (innerMatches && innerMatches.length) pending.push(...innerMatches);
          });
        }
        if (pending.length === 0) return;
        // Process in a microtask to batch DOM work
        requestAnimationFrame(() => {
          pending.forEach((el) => this.filterFacebookReply(el));
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Also process any existing replies on load
      document.querySelectorAll(selector).forEach((el) => this.filterFacebookReply(el));
      console.log('JeetBlock: Facebook replies observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook replies observer', e);
    }
  }

  // Process a single Facebook reply node in a strictly scoped way
  filterFacebookReply(replyEl) {
    try {
      if (!this.isEnabled || !this.filterFacebook) return;
      if (!replyEl || replyEl.dataset.harmonizerProcessed === 'true') return;

      // Mark processed early to avoid duplicate handling
      replyEl.dataset.harmonizerProcessed = 'true';

      // Extract username primarily from aria-label: "Comment by <name> ..."
      const aria = replyEl.getAttribute('aria-label') || '';
      let username = '';
      const byIdx = aria.indexOf('Comment by ');
      if (byIdx >= 0) {
        const after = aria.slice(byIdx + 'Comment by '.length).trim();
        // Heuristic: cut before the time chunk like "20 weeks ago" (find ' ago')
        const agoIdx = after.indexOf(' ago');
        if (agoIdx > 0) {
          username = after.slice(0, agoIdx).trim();
        } else {
          // Fallback: cut at two-space transition or end
          const parts = after.split(/\s\d/);
          username = (parts[0] || after).trim();
        }
      }

      // Fallbacks from visible link text near header
      if (!username) {
        const nameLink = replyEl.querySelector('a[role="link"][href*="profile.php"], a[role="link"][href^="/people/"], a[role="link"][href^="/"]');
        if (nameLink && nameLink.textContent) username = nameLink.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: FB-Comment: No username found — skipped');
        return;
      }

      // Extract first visible text line for content-based checks
      const contentNode = replyEl.querySelector('div[dir="auto"]');
      const contentText = contentNode ? (contentNode.textContent || '').trim() : '';

      const textLower = (username + ' ' + contentText).toLowerCase();
      const usernameLower = (username || '').toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering: NAME checks on username only; scripts/emojis may use content
      if (this.filterIndian) {
        const indianReasons = [];
        if (this.containsIndianName(usernameLower)) indianReasons.push('Indian name');
        if (this.containsDevanagari(textLower) || this.containsBengali(textLower) || this.containsSinhala(textLower) || this.containsGurmukhi(textLower)) indianReasons.push('Indian script');
        if (this.containsIndianEmoji(username) || this.containsIndianEmoji(contentText)) indianReasons.push('Indian emoji');
        if (indianReasons.length) { shouldFilter = true; reasons.push(...indianReasons); }
      }

      // Muslim filtering (scripts + names + emojis) — Tajik/Turkish/Uzbek disabled globally
      if (!shouldFilter && this.filterMuslim) {
        const muslimReasons = [];
        if (this.containsArabic(textLower)) muslimReasons.push('Arabic script');
        if (this.containsPersian(textLower)) muslimReasons.push('Persian script');
        if (this.containsUrdu(textLower)) muslimReasons.push('Urdu script');
        const muslimMatchedNames = this.getMatchedMuslimNames(usernameLower); // names on username only
        if (muslimMatchedNames.length) muslimReasons.push(`Muslim name [${muslimMatchedNames.join(', ')}]`);
        if (this.containsMuslimEmoji(username) || this.containsMuslimEmoji(contentText)) muslimReasons.push('Muslim emoji');
        if (muslimReasons.length > 0) {
          shouldFilter = true;
          reasons.push(...muslimReasons);
        }
      }

      // Jewish filtering: NAME checks on username only; scripts/emojis may use content
      if (!shouldFilter && this.filterJewish) {
        const jewishReasons = [];
        if (this.containsJewishName(usernameLower)) jewishReasons.push('Jewish name');
        if (this.containsHebrew(textLower)) jewishReasons.push('Hebrew script');
        if (this.containsJewishEmoji(username) || this.containsJewishEmoji(contentText)) jewishReasons.push('Jewish emoji');
        if (jewishReasons.length) { shouldFilter = true; reasons.push(...jewishReasons); }
      }

      if (!shouldFilter) {
        // Clear processed flag so future setting changes can re-process
        delete replyEl.dataset.harmonizerProcessed;
        return;
      }

      console.log(`JeetBlock: FB-Comment: Filtering reply by ${username} (reasons: ${reasons.join(', ')})`);
      // Tag as facebook for styling consistency
      replyEl.dataset.facebook = 'true';
      this.applyFilter(replyEl);
    } catch (e) {
      console.warn('JeetBlock: FB-Comment: Error filtering reply', e);
    }
  }

  // ========================= Facebook Messages (Scoped) =========================
  getFacebookMessageSelector() {
    // Anchor that wraps each thread row in the messages list
    return [
      'div[role="row"] a[href^="/messages/"]',
      '[data-pagelet="MWThreadList"] a[role="link"]',
      'div[aria-label="Chats"] [role="link"]',
      'ul[role="grid"] [role="row"] a[role="link"]'
    ].join(', ');
  }

  setupFacebookMessagesObserver() {
    try {
      const selector = this.getFacebookMessageSelector();
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const mutation of mutations) {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const innerMatches = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (innerMatches && innerMatches.length) pending.push(...innerMatches);
          });
        }
        if (pending.length === 0) return;
        requestAnimationFrame(() => {
          pending.forEach((el) => this.filterFacebookMessage(el));
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process any existing message threads on load
      document.querySelectorAll(selector).forEach((el) => this.filterFacebookMessage(el));
      console.log('JeetBlock: Facebook messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook messages observer', e);
    }
  }

  filterFacebookMessage(messageEl) {
    try {
      console.log('JeetBlock: filterFacebookMessage called on:', messageEl);
      if (!this.isEnabled || !this.filterFacebook) {
        console.log('JeetBlock: Facebook message filtering disabled. Enabled:', this.isEnabled, 'FilterFacebook:', this.filterFacebook);
        return;
      }
      if (!messageEl || messageEl.dataset.harmonizerProcessed === 'true') return;

      // Username extraction — prefer the visible name span inside the thread row
      let username = '';
      // Heuristic 1: first visible name-like span
      const nameCandidates = messageEl.querySelectorAll('div[dir="ltr"] span, [dir="auto"] span, strong');
      for (const span of nameCandidates) {
        const txt = this.getTextWithEmojis(span);
        if (!txt) continue;
        // Skip timestamps/separators like ".", "·", "11m", "2h", "ago"
        if (/^·$/.test(txt) || /\b\d+\s*[smhdw]\b/i.test(txt) || /\bago\b/i.test(txt)) continue;
        username = txt;
        break;
      }

      // Fallback: aria-label (split before separators)
      if (!username) {
        const aria = messageEl.getAttribute && messageEl.getAttribute('aria-label');
        if (aria && aria.trim()) {
          const namePart = aria.split('·')[0].split('•')[0].split('|')[0].trim();
          if (namePart) username = namePart;
        }
      }

      if (!username) {
        console.log('JeetBlock: FB-Message: No username found — skipped');
        return;
      }

      // Message preview text (exclude timestamps and separators)
      let contentText = '';
      const contentSpans = messageEl.querySelectorAll('span[dir="auto"]');
      for (const span of contentSpans) {
        const text = this.getTextWithEmojis(span);
        if (!text) continue;
        if (/\b\d+\s*m\b|\b\d+\s*h\b|\bago\b|^·$/.test(text)) continue;
        contentText += text + ' ';
      }
      contentText = contentText.trim();

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
      if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
      if (this.containsDevanagari(fullTextLower)) { shouldFilter = true; reasons.push('Devanagari script'); }
      if (this.containsBengali(fullTextLower)) { shouldFilter = true; reasons.push('Bengali script'); }
      if (this.containsSinhala(fullTextLower)) { shouldFilter = true; reasons.push('Sinhala script'); }
      if (this.containsGurmukhi(fullTextLower)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
      if (this.containsIndianEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(fullTextLower)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(fullTextLower)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(fullTextLower)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(fullTextLower)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(fullTextLower)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(fullTextLower)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(fullTextLower)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(fullTextLower)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) {
        // allow future re-processing if settings change (do not set processed)
        return;
      }

      // Mark processed only when we actually filter
      messageEl.dataset.harmonizerProcessed = 'true';
      messageEl.dataset.fbMessage = 'true';
      this.applyFilter(messageEl, 'facebook-message');
      console.log(`JeetBlock: FB-Message: Filtering message from ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: FB-Message error', e);
    }
  }

  // When a filtered message is opened, collapse the message thread grid too
  setupFacebookMessageThreadObserver() {
    try {
      const selector = 'div[role="grid"][aria-label^="Messages in conversation with"], div[role="gridcell"][data-scope="messages_table"]';
      const observer = new MutationObserver((mutations) => {
        const grids = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) grids.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => grids.add(el));
          });
        }
        if (grids.size === 0) return;
        requestAnimationFrame(() => {
          grids.forEach((grid) => this.filterFacebookMessageThread(grid));
        });
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((grid) => this.filterFacebookMessageThread(grid));
      console.log('JeetBlock: Facebook message thread observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook message thread observer', e);
    }
  }

  filterFacebookMessageThread(gridEl) {
    try {
      if (!this.isEnabled || !this.filterFacebook) return;
      if (!gridEl || gridEl.dataset.harmonizerProcessed === 'true') return;

      // Find the header name inside the thread view
      const nameSpan = document.querySelector('h2 span.xlyipyv.xuxw1ft, h2 [dir="auto"] span');
      const username = nameSpan ? (nameSpan.textContent || '').trim() : '';
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(username)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(username)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(username)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(username)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(username)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(username)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(username)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      gridEl.dataset.harmonizerProcessed = 'true';
      gridEl.dataset.fbMessageThread = 'true';
      this.applyFilter(gridEl, 'facebook-message-thread');
      console.log(`JeetBlock: FB-Thread: Collapsed thread for ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: FB-Thread error', e);
    }
  }

  // ========================= Facebook Sidebar Messages =========================
  setupFacebookSidebarMessagesObserver() {
    try {
      const selector = 'li a[href*="/messages/"]';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterFacebookSidebarMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterFacebookSidebarMessage(item));
      console.log('JeetBlock: Facebook sidebar messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook sidebar messages observer', e);
    }
  }

  filterFacebookSidebarMessage(messageItem) {
    if (!this.isEnabled || !this.filterFacebook) return;
    try {
      if (!messageItem || messageItem.dataset.harmonizerProcessed === 'true') return;

      // Extract username from the span with dir="auto"
      const usernameEl = messageItem.querySelector('span[dir="auto"]');
      if (!usernameEl) return;

      const username = usernameEl.textContent.trim();
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(username)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(username)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(username)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(username)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(username)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(username)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(username)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      messageItem.dataset.harmonizerProcessed = 'true';
      messageItem.dataset.fbSidebarMessage = 'true';
      messageItem.dataset.fbFilteredUsername = username; // Store username for content linking
      this.applyFilter(messageItem, 'facebook-sidebar-message');
      
      // Also filter any existing message content for this user
      this.filterFacebookMessagesByUsername(username);
      
      console.log(`JeetBlock: Facebook Sidebar Message filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Facebook Sidebar Message filter error', e);
    }
  }

  // Helper method to filter Facebook messages by username
  filterFacebookMessagesByUsername(username) {
    if (!username) return;
    
    // Find all message elements that might contain content from this user
    const messageSelectors = [
      'div[role="grid"][aria-label^="Messages in conversation with"]',
      'div[role="gridcell"][data-scope="messages_table"]',
      'div[data-scope="messages_table"]'
    ];
    
    messageSelectors.forEach(selector => {
      const elements = document.querySelectorAll(selector);
      elements.forEach(el => {
        if (el.dataset.harmonizerProcessed === 'true') return;
        
        // Check if this message element contains content from the filtered user
        const nameSpans = el.querySelectorAll('span[dir="auto"], div[dir="ltr"] span, [dir="auto"] span, strong');
        for (const span of nameSpans) {
          const spanText = this.getTextWithEmojis(span);
          if (spanText === username) {
            // This is a message from the filtered user, filter it
            el.dataset.harmonizerProcessed = 'true';
            el.dataset.fbMessageContent = 'true';
            this.applyFilter(el, 'facebook-message-content');
            console.log(`JeetBlock: Facebook Message Content filtered for user: ${username}`);
            break;
          }
        }
      });
    });
  }

  // ========================= Facebook Suggested Friends (Scoped) =========================
  setupFacebookSuggestedFriendsObserver() {
    try {
      // Selector targets links to suggested friend profiles
      const selector = 'a[href*="/friends/suggestions/?profile_id="]';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterFacebookSuggestedFriend(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterFacebookSuggestedFriend(item));
      console.log('JeetBlock: Facebook suggested friends observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook suggested friends observer', e);
    }
  }

  filterFacebookSuggestedFriend(linkEl) {
    if (!this.isEnabled || !this.filterFacebook) return;
    try {
      if (!linkEl || linkEl.dataset.harmonizerProcessed === 'true') return;

      // Extract username from link text content
      let username = linkEl.textContent.trim();
      
      // If link text is empty, try to get from nearby img alt text
      if (!username) {
        const img = linkEl.querySelector('img[alt*="Profile Photo of"]');
        if (img && img.alt) {
          const match = img.alt.match(/Profile Photo of (.+)/);
          if (match) username = match[1].trim();
        }
      }

      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Muslim name/script filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
      }

      // Jewish name/script filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // Indian name/script filtering
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      // Find the card container to hide (parent of the link)
      let cardEl = linkEl.closest('[data-type="hscroll-child"]');
      if (!cardEl) cardEl = linkEl.closest('.x1c4vz4f');
      if (!cardEl) cardEl = linkEl;

      cardEl.dataset.harmonizerProcessed = 'true';
      cardEl.dataset.fbSuggestedFriend = 'true';
      this.applyFilter(cardEl, 'facebook-suggested-friend');
      console.log(`JeetBlock: Facebook Suggested Friend filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Facebook Suggested Friend filter error', e);
    }
  }

  // ========================= Facebook Reels (Scoped) =========================
  setupFacebookReelsObserver() {
    try {
      // Selector targets reel links
      const selector = 'a[aria-label="reel"][href*="/reel/"]';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterFacebookReel(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterFacebookReel(item));
      console.log('JeetBlock: Facebook reels observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Facebook reels observer', e);
    }
  }

  filterFacebookReel(reelLink) {
    if (!this.isEnabled || !this.filterFacebook) return;
    try {
      if (!reelLink || reelLink.dataset.harmonizerProcessed === 'true') return;

      console.log('JeetBlock: FB-Reel: Processing reel element:', reelLink);
      console.log('JeetBlock: FB-Reel: Link href:', reelLink.href);

      // Try multiple methods to extract username
      let username = '';
      
      // Method 1: Look for aria-label on link
      const ariaLabel = reelLink.getAttribute('aria-label');
      console.log('JeetBlock: FB-Reel: aria-label:', ariaLabel);
      
      // Method 2: Look for text in nearby siblings or parent
      const parent = reelLink.parentElement;
      if (parent) {
        console.log('JeetBlock: FB-Reel: Parent element:', parent);
        console.log('JeetBlock: FB-Reel: Parent textContent:', parent.textContent.substring(0, 200));
        
        // Try to find username in sibling elements
        const siblings = parent.parentElement?.children || [];
        for (let sibling of siblings) {
          if (sibling !== parent && sibling.textContent && sibling.textContent.trim()) {
            console.log('JeetBlock: FB-Reel: Sibling text:', sibling.textContent.substring(0, 100));
          }
        }
      }
      
      // Method 3: Look for span or div with username-like classes near the reel
      const container = reelLink.closest('[data-type="hscroll-child"]');
      if (container) {
        console.log('JeetBlock: FB-Reel: Container found');
        const allText = container.textContent.trim();
        console.log('JeetBlock: FB-Reel: Container full text:', allText.substring(0, 200));
        
        // Try to find spans with text
        const spans = container.querySelectorAll('span[dir="auto"], a[role="link"]');
        spans.forEach((span, idx) => {
          const text = this.getTextWithEmojis(span);
          if (text && text.length > 0 && text.length < 100) {
            console.log(`JeetBlock: FB-Reel: Span ${idx} text:`, text);
          }
        });
      }
      
      // Method 4: Look in the hscroll container for adjacent text
      const hscrollParent = reelLink.closest('[data-type="hscroll-child"]')?.parentElement;
      if (hscrollParent) {
        const reelCards = hscrollParent.querySelectorAll('[data-type="hscroll-child"]');
        console.log('JeetBlock: FB-Reel: Found', reelCards.length, 'reel cards in hscroll');
      }

      if (!username) {
        console.log('JeetBlock: FB-Reel: No username found - cannot filter yet');
        reelLink.dataset.harmonizerProcessed = 'true'; // Mark as processed to avoid re-checking
        return;
      }

      console.log('JeetBlock: FB-Reel: Extracted username:', username);

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Apply all filter checks
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
      }
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      // Find the card container to hide
      let cardEl = reelLink.closest('[data-type="hscroll-child"]');
      if (!cardEl) cardEl = reelLink.closest('.x1c4vz4f');
      if (!cardEl) cardEl = reelLink;

      cardEl.dataset.harmonizerProcessed = 'true';
      cardEl.dataset.fbReel = 'true';
      this.applyFilter(cardEl, 'facebook-reel');
      console.log(`JeetBlock: Facebook Reel filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Facebook Reel filter error', e);
    }
  }

  // ========================= Facebook Chat Bubble (Stub) =========================
  setupFacebookChatBubbleObserver() {
    // Placeholder for future implementation
    console.log('JeetBlock: Facebook chat bubble observer (not yet implemented)');
  }

  // ========================= Twitter Messages (Scoped) =========================
  getTwitterMessageSelector() {
    // Conversation link in DM list
    return 'div[role="link"][data-testid="conversation"]';
  }

  setupTwitterMessagesObserver() {
    try {
      const selector = this.getTwitterMessageSelector();
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const mutation of mutations) {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) pending.push(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (inner && inner.length) pending.push(...inner);
          });
        }
        if (pending.length === 0) return;
        requestAnimationFrame(() => pending.forEach((el) => this.filterTwitterMessage(el)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((el) => this.filterTwitterMessage(el));
      console.log('JeetBlock: Twitter messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Twitter messages observer', e);
    }
  }

  filterTwitterMessage(convEl) {
    try {
      console.log('JeetBlock: filterTwitterMessage called on:', convEl, 'Has dataset?', convEl?.dataset, 'Already processed?', convEl?.dataset?.harmonizerProcessed);
      if (!this.isEnabled || !this.filterTwitter) {
        console.log('JeetBlock: Twitter message filtering disabled. Enabled:', this.isEnabled, 'FilterTwitter:', this.filterTwitter);
        return;
      }
      if (!convEl || convEl.dataset.harmonizerProcessed === 'true') {
        console.log('JeetBlock: Message skipped - null element or already processed');
        return;
      }

      // Extract visible display name within the conversation item
      let username = '';
      // Prefer display name near the conversation row
      const display1 = convEl.querySelector('div[dir="ltr"] span');
      if (display1) username = this.getTextWithEmojis(display1);

      // Fallback: search a bit more broadly within the row
      if (!username) {
        const display2 = convEl.querySelector('[dir="auto"] span, [role="link"] [dir="ltr"] span');
        if (display2) username = this.getTextWithEmojis(display2);
      }

      // Fallback: climb to ancestor activeRoute and read its visible span
      if (!username) {
        const route = convEl.closest('[data-testid="activeRoute"]');
        const display3 = route ? route.querySelector('div[dir="ltr"] span, [dir="auto"] span') : null;
        if (display3) username = this.getTextWithEmojis(display3);
      }

      if (!username) {
        console.log('JeetBlock: TW-Message: No username found — skipped');
        // Do not mark as processed so we can retry on future mutations
        return;
      }

      // Message preview text
      let contentText = '';
      const preview = convEl.querySelector('[data-testid="tweetText"], div[dir="ltr"]');
      if (preview) contentText = this.getTextWithEmojis(preview);

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
      if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
      if (this.containsDevanagari(fullTextLower)) { shouldFilter = true; reasons.push('Devanagari script'); }
      if (this.containsBengali(fullTextLower)) { shouldFilter = true; reasons.push('Bengali script'); }
      if (this.containsSinhala(fullTextLower)) { shouldFilter = true; reasons.push('Sinhala script'); }
      if (this.containsGurmukhi(fullTextLower)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
      if (this.containsIndianEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(fullTextLower)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(fullTextLower)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(fullTextLower)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(fullTextLower)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(fullTextLower)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(fullTextLower)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(fullTextLower)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(fullTextLower)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(fullTextLower)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) {
        // Not filtered – leave unprocessed so future updates can retry
        return;
      }

      // Mark processed only when we actually filter
      convEl.dataset.harmonizerProcessed = 'true';
      convEl.dataset.twMessage = 'true';
      this.applyFilter(convEl, 'twitter-message');
      console.log(`JeetBlock: TW-Message: Filtering message from ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: TW-Message error', e);
    }
  }

  // ========================= Twitter Main Messaging Window =========================
  setupTwitterMainMessagingObserver() {
    try {
      // Target the conversation container and individual message entries
      const conversationSelector = 'div[data-testid="DmActivityContainer"]';
      const messageSelector = 'button[data-testid="messageEntry"]';
      
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(conversationSelector)) items.add(node);
            if (node.matches && node.matches(messageSelector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(conversationSelector + ',' + messageSelector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterTwitterMainMessagingMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(conversationSelector + ',' + messageSelector).forEach((item) => this.filterTwitterMainMessagingMessage(item));
      console.log('JeetBlock: Twitter main messaging observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Twitter main messaging observer', e);
    }
  }

  filterTwitterMainMessagingMessage(element) {
    if (!this.isEnabled || !this.filterTwitter) return;
    try {
      if (!element || element.dataset.harmonizerProcessed === 'true') return;

      // Skip if this is part of the message input area
      if (element.closest('aside[aria-label="Start a new message"]')) return;

      let username = '';
      let targetElement = element;

      // Check if this is a conversation container
      if (element.matches && element.matches('div[data-testid="DmActivityContainer"]')) {
        // For conversation containers, get username from header
        const headerEl = element.querySelector('h2[aria-level="2"] span span');
        if (headerEl) {
          username = headerEl.textContent.trim();
          targetElement = element; // Filter the entire conversation
        }
      } else if (element.matches && element.matches('button[data-testid="messageEntry"]')) {
        // For individual messages, try to get username from conversation header
        const conversationContainer = element.closest('div[data-testid="DmActivityContainer"]');
        if (conversationContainer) {
          const headerEl = conversationContainer.querySelector('h2[aria-level="2"] span span');
          if (headerEl) {
            username = headerEl.textContent.trim();
            targetElement = conversationContainer; // Filter the entire conversation
          }
        }
      }

      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(username)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(username)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(username)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(username)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(username)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(username)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(username)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      targetElement.dataset.harmonizerProcessed = 'true';
      targetElement.dataset.twitterMainMessagingMessage = 'true';
      this.applyFilter(targetElement, 'twitter-main-messaging-message');
      console.log(`JeetBlock: Twitter Main Messaging Conversation filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Twitter Main Messaging Message filter error', e);
    }
  }

  // ========================= Twitter Conversation Details Pane =========================
  setupTwitterConversationDetailsObserver() {
    try {
      // Section containing the conversation details (header + scroller)
      const selector = 'section[aria-label="Section details"]';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterTwitterConversationDetails(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterTwitterConversationDetails(item));
      console.log('JeetBlock: Twitter conversation details observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize Twitter conversation details observer', e);
    }
  }

  filterTwitterConversationDetails(sectionEl) {
    if (!this.isEnabled || !this.filterTwitter) return;
    try {
      if (!sectionEl || sectionEl.dataset.harmonizerProcessed === 'true') return;

      // Get username from the header path shown in the provided outerHTML
      let username = '';
      const headerName = sectionEl.querySelector('#detail-header span span');
      if (headerName) username = headerName.textContent.trim();

      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Indian filtering
      if (this.filterIndian) {
        if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }
        if (this.containsDevanagari(username)) { shouldFilter = true; reasons.push('Devanagari script'); }
        if (this.containsBengali(username)) { shouldFilter = true; reasons.push('Bengali script'); }
        if (this.containsSinhala(username)) { shouldFilter = true; reasons.push('Sinhala script'); }
        if (this.containsGurmukhi(username)) { shouldFilter = true; reasons.push('Gurmukhi script'); }
        if (this.containsIndianEmoji(username)) { shouldFilter = true; reasons.push('Indian emoji'); }
      }

      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
        if (this.containsArabic(username)) { shouldFilter = true; reasons.push('Arabic script'); }
        if (this.containsPersian(username)) { shouldFilter = true; reasons.push('Persian script'); }
        if (this.containsUrdu(username)) { shouldFilter = true; reasons.push('Urdu script'); }
        if (this.containsIndonesian(username)) { shouldFilter = true; reasons.push('Indonesian words'); }
        if (this.containsMalay(username)) { shouldFilter = true; reasons.push('Malay words'); }
        if (this.containsPashto(username)) { shouldFilter = true; reasons.push('Pashto script'); }
        if (this.containsDari(username)) { shouldFilter = true; reasons.push('Dari script'); }
        if (this.containsMuslimEmoji(username)) { shouldFilter = true; reasons.push('Muslim emoji'); }
      }

      // Jewish/Israeli filtering
      if (this.filterJewish) {
        if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
        if (this.containsHebrew(username)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (this.containsJewishEmoji(username)) { shouldFilter = true; reasons.push('Jewish emoji'); }
      }

      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(username)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }

      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }

      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(username)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }

      // Custom filters
      if (this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      // Hide the scroller container to collapse the whole pane contents
      const scroller = sectionEl.querySelector('[data-testid="DmScrollerContainer"]') || sectionEl;
      scroller.dataset.harmonizerProcessed = 'true';
      scroller.dataset.twitterConversationDetails = 'true';
      this.applyFilter(scroller, 'twitter-conversation-details');
      console.log(`JeetBlock: Twitter Conversation Details filtered: ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: Twitter Conversation Details filter error', e);
    }
  }





  getPlatform() {
    const host = location.host;
    if (host.includes('twitter.com') || host.includes('x.com')) return 'twitter';
    if (host.includes('facebook.com') || host.includes('messenger.com')) return 'facebook';
    if (host.includes('instagram.com')) return 'instagram';
    if (host.includes('linkedin.com')) return 'linkedin';
    if (host.includes('tiktok.com')) return 'tiktok';
    if (host.includes('reddit.com')) return 'reddit';
    if (host.includes('youtube.com')) return 'youtube';
    return 'unknown';
  }

  async loadSettings() {
    const result = await chrome.storage.sync.get({
      filterMode: 'collapse',
      isEnabled: true,
      customFilters: [],
      allowList: [],
      blockList: [],
      filterIndian: true,
      filterMuslim: true,
      filterJewish: true,
      filterLGBTQIA: true,
      filterUkraine: true,
      filterPalestine: true,
      // Platform-specific filtering
      filterFacebook: true,
      filterTwitter: true,
      filterInstagram: true,
      filterLinkedIn: true,
      filterTikTok: true,
      filterReddit: true,
      filterYouTube: true
    });
    
    this.filterMode = result.filterMode;
    this.isEnabled = result.isEnabled;
    this.customFilters = result.customFilters;
    this.allowList = result.allowList;
    this.blockList = result.blockList;
    this.filterIndian = result.filterIndian;
    this.filterMuslim = result.filterMuslim;
    this.filterJewish = result.filterJewish;
    this.filterLGBTQIA = result.filterLGBTQIA;
    this.filterUkraine = result.filterUkraine;
    this.filterPalestine = result.filterPalestine;
    // Platform-specific filtering
    this.filterFacebook = result.filterFacebook;
    this.filterTwitter = result.filterTwitter;
    this.filterInstagram = result.filterInstagram;
    this.filterLinkedIn = result.filterLinkedIn;
    this.filterTikTok = result.filterTikTok;
    this.filterReddit = result.filterReddit;
    this.filterYouTube = result.filterYouTube;
    
    console.log('JeetBlock: Settings loaded in content script:', {
      isEnabled: this.isEnabled,
      allowList: this.allowList,
      blockList: this.blockList,
      filterIndian: this.filterIndian,
      filterMuslim: this.filterMuslim,
      filterJewish: this.filterJewish,
      filterLGBTQIA: this.filterLGBTQIA,
      filterUkraine: this.filterUkraine,
      filterPalestine: this.filterPalestine,
      filterFacebook: this.filterFacebook,
      filterTwitter: this.filterTwitter,
      filterInstagram: this.filterInstagram,
      filterLinkedIn: this.filterLinkedIn,
      filterTikTok: this.filterTikTok,
      filterReddit: this.filterReddit,
      filterYouTube: this.filterYouTube
    });
    
    console.log('JeetBlock: LinkedIn filter status:', this.filterLinkedIn);
  }

  async loadIndianNames() {
    try {
      const response = await fetch(chrome.runtime.getURL('indianNames.json'));
      const data = await response.json();
      this.indianNames = data.names || [];
    } catch (error) {
      console.log('Could not load Indian names, using fallback list');
      this.indianNames = [
        'Aarav', 'Aarush', 'Advait', 'Arjun', 'Arnav', 'Aryan', 'Dhruv', 'Ishaan',
        'Krishna', 'Lakshay', 'Neel', 'Reyansh', 'Shaurya', 'Vivaan', 'Yash',
        'Aanya', 'Ananya', 'Diya', 'Ira', 'Kiara', 'Mira', 'Myra', 'Navya',
        'Pari', 'Riya', 'Saanvi', 'Sara', 'Tara', 'Zara', 'Zoya'
      ];
    }
  }

  async loadMuslimNames() {
    try {
      const response = await fetch(chrome.runtime.getURL('muslimNames.json'));
      const data = await response.json();
      this.muslimNames = data.names || [];
    } catch (error) {
      console.log('Could not load Muslim names, using fallback list');
      this.muslimNames = [
        'Ahmad', 'Ahmed', 'Ali', 'Amir', 'Asad', 'Asif', 'Aziz', 'Bilal',
        'Fahad', 'Farhan', 'Fazal', 'Hamza', 'Hassan', 'Husain', 'Hussein', 'Ibrahim',
        'Imran', 'Irfan', 'Ishaq', 'Ismail', 'Jabir', 'Jafar', 'Jamil', 'Kamal',
        'Karim', 'Khalid', 'Khalil', 'Mahmood', 'Mahmud', 'Malik', 'Mansoor', 'Mansur',
        'Masood', 'Masud', 'Mehdi', 'Muhammad', 'Mohammad', 'Mohammed', 'Murtaza', 'Musa',
        'Mustafa', 'Nadeem', 'Nadir', 'Naeem', 'Najeeb', 'Naseem', 'Nasir', 'Nawaz',
        'Nazir', 'Omar', 'Omer', 'Qasim', 'Qays', 'Rafiq', 'Rahim', 'Rahman',
        'Rashid', 'Raza', 'Rehman', 'Saeed', 'Saif', 'Saleem', 'Salim', 'Samir',
        'Saqib', 'Sarfraz', 'Shahid', 'Shahzad', 'Shakir', 'Shamshad', 'Sharif', 'Shaukat',
        'Shoaib', 'Sohail', 'Suleman', 'Sulaiman', 'Tahir', 'Tariq', 'Umar', 'Usman',
        'Wahid', 'Wajid', 'Waqar', 'Waseem', 'Wasim', 'Yasir', 'Younis', 'Yusuf',
        'Zafar', 'Zahid', 'Zain', 'Zakir', 'Zaman', 'Zeeshan', 'Zubair', 'Zulfiqar'
      ];
    }
  }

  async loadJewishNames() {
    try {
      const response = await fetch(chrome.runtime.getURL('jewishNames.json'));
      const data = await response.json();
      this.jewishNames = data.names || [];
      console.log('JeetBlock: Loaded Jewish names:', this.jewishNames.length);
    } catch (error) {
      console.error('JeetBlock: Failed to load Jewish names from JSON:', error);
      console.log('JeetBlock: Using fallback list instead');
      this.jewishNames = [
        'Cohen', 'Levy', 'Levin', 'Levine', 'Goldberg', 'Goldman', 'Goldstein', 'Silverman',
        'Silverstein', 'Rosenberg', 'Rosen', 'Rosenstein', 'Roth', 'Rothman', 'Rothstein',
        'Weinberg', 'Weinstein', 'Weiss', 'Weissman', 'Weissmann', 'Schwartz', 'Schwarz',
        'Schwarzman', 'Schwarzmann', 'Greenberg', 'Green', 'Greene', 'Greenman', 'Greenstein',
        'Brownstein', 'Blackman', 'Blackstein', 'Whitman',
        'Whitstein', 'Klein', 'Kleinfeld', 'Kleinman', 'Kleinstein', 'Gross', 'Grossman',
        'Grossberg', 'Grossstein', 'Friedman', 'Freedman', 'Freedberg', 'Freedstein',
        'Katz', 'Katzman', 'Katzberg', 'Katzstein', 'Millman', 'Millberg',
        'Millstein', 'Solomon', 'Solomonov',
        'Solomons', 'Solomonberg', 'Solomonstein', 'Jacobson', 'Jacobs', 'Jacobberg',
        'Jacobstein', 'Isaacson', 'Isaacs', 'Isaacberg', 'Isaacstein', 'Abraham',
        'Abrams', 'Abramson', 'Abramberg', 'Abramstein', 'Moses', 'Moskowitz', 'Moskowitzberg',
        'Moskowitzstein', 'Aarons', 'Aaronson', 'Aaronberg', 'Aaronstein',
        'Raphaels', 'Raphaelson', 'Raphaelberg', 'Raphaelstein',
        'Simons', 'Simonson', 'Simonberg',
        'Simonstein',
      ];
    }
  }

  getPostSelectors() {
    const selectors = {
      twitter: '[data-testid="tweet"], div[role="link"][data-testid="conversation"], div[data-testid="DmActivityContainer"], section[aria-label="Section details"], button[data-testid="UserCell"]',
      facebook: '[role="article"], [data-pagelet="FeedUnit_0"], .x1yztbdb, .x1n2onr6[data-pagelet], [data-video-id], [data-type="hscroll-child"], .xfpmyvw, [data-pagelet="MessengerChatTab"], [data-pagelet="MessengerConversation"], .html-div.xdj266r.x14z9mp.xat24cr.x1lziwak.xexx8yu.xyri2b.x18d9i69.x1c1uobl.x78zum5.x1eb86dx',
      instagram: 'article:not(.x7goit8), [role="article"]:not(.x7goit8), .x78zum5.xdt5ytf.x5yr21d.xa1mljc.xh8yej3:not(.x7goit8)',
      linkedin: 'div.feed-shared-update-v2[role="article"]',
      tiktok: 'article[data-e2e="recommend-list-item-container"], div[data-e2e="explore-item"], div[data-e2e="explore-item"] + div[data-e2e="explore-card-desc"]',
      reddit: 'article[aria-label], shreddit-post, [data-testid="post-container"], .Post, [data-click-id="body"], .thing',
      youtube: '' // YouTube elements are handled by dedicated observers, not the main filtering logic
    };
    return selectors[this.platform] || '[role="article"], [data-testid="tweet"]';
  }

  getUsernameSelectors() {
    const selectors = {
      twitter: '[data-testid="User-Name"] a, [data-testid="User-Name"] span',
      facebook: 'a[aria-label*=" "], h3 a, [data-ad-rendering-role="profile_name"] a, [data-ad-rendering-role="profile_name"] span, a[href*="/facebook.com/"], [aria-label*="· Original audio"], [aria-label*="·"], h2, .xfpmyvw h2, .xfpmyvw span, [data-pagelet="MessengerChatTab"] h2, [data-pagelet="MessengerChatTab"] span, [data-pagelet="MessengerConversation"] h2, [data-pagelet="MessengerConversation"] span, img[alt*=" "], .html-div.xdj266r.x14z9mp.xat24cr.x1lziwak.xexx8yu.xyri2b.x18d9i69.x1c1uobl.x78zum5.x1eb86dx img[alt]',
      instagram: 'a[href*="/"][role="link"]:not([href*="/p/"]):not([href*="/reels/"]):not([href*="/explore/"]):not([href*="/stories/"]), img[alt*="profile picture"]',
      linkedin: 'a[aria-label]',
      tiktok: 'a[href*="/@"], [data-e2e="video-author-uniqueid"], [data-e2e="video-author-avatar"], [data-e2e="explore-card-user-link"], [data-e2e="explore-card-user-unique-id"]',
      reddit: 'a[href*="/user/"], a[href*="/u/"], [slot="authorName"] a, .author, .username',
      youtube: '#author-text, .shortsLockupViewModelHostMetadataTitle a, .shortsLockupViewModelHostOutsideMetadataTitle a, ytd-channel-name a, ytd-video-meta-block #channel-name a, ytd-video-meta-block #owner-name a, .ytd-video-meta-block #channel-name a, .ytd-video-meta-block #owner-name a, #owner-name a, #channel-name a, .ytd-video-meta-block a, .ytd-video-meta-block span, .ytd-video-meta-block yt-formatted-string, ytd-video-owner-renderer a, ytd-video-owner-renderer yt-formatted-string, ytd-watch-metadata yt-formatted-string, ytd-watch-metadata a[href*="/@"], .yt-core-attributed-string'
    };
    return selectors[this.platform] || 'a[href*="/"]';
  }

  getContentSelectors() {
    const selectors = {
      twitter: '[data-testid="tweetText"]',
      facebook: '[data-testid="post_message"], .userContent, [data-ad-rendering-role="post_message"], .x1iorvi4, .x1n2onr6[dir="auto"]',
      instagram: 'div[data-testid="post-caption"], ._ap3a._aaco._aacu._aacx._aad7._aade, [class*="_aacu"], span[dir="auto"]',
      linkedin: '.feed-shared-text, .update-components-text, .feed-shared-inline-show-more-text, .update-components-update-v2__commentary',
      tiktok: '[data-e2e="video-desc"], img[alt*="created by"]',
      reddit: '[slot="title"], [slot="text-body"], .condensed-post-title-heading, .post-content, .usertext-body, .entry .md',
      youtube: 'yt-attributed-string#content-text, .yt-core-attributed-string, .shortsLockupViewModelHostMetadataTitle, .shortsLockupViewModelHostOutsideMetadataTitle'
    };
    return selectors[this.platform] || 'div, span, p';
  }

  // Helper function to check if text only contains common punctuation
  isOnlyCommonPunctuation(text) {
    if (!text) return false;
    // Remove common punctuation and check if anything remains
    const cleaned = text.replace(/[!\-|+=_,.\s]/g, '');
    return cleaned.length === 0;
  }

  // Helper function to extract text including emoji alt attributes
  getTextWithEmojis(element) {
    if (!element) return '';
    
    let text = '';
    
    // Walk through all child nodes
    for (const node of element.childNodes) {
      if (node.nodeType === Node.TEXT_NODE) {
        text += node.textContent;
      } else if (node.nodeType === Node.ELEMENT_NODE) {
        // Check if it's an img with alt text (emoji)
        if (node.tagName === 'IMG' && node.alt) {
          text += node.alt;
        } else {
          // Recursively get text from child elements
          text += this.getTextWithEmojis(node);
        }
      }
    }
    
    return text.trim();
  }

  containsDevanagari(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const devanagariRegex = /[\u0900-\u097F]/;
    return devanagariRegex.test(text);
  }

  containsArabic(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const arabicRegex = /[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFF]/;
    return arabicRegex.test(text);
  }

  containsPersian(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const persianRegex = /[\u0600-\u06FF\uFB50-\uFDFF\uFE70-\uFEFF]/;
    return persianRegex.test(text);
  }

  containsTurkish(text) {
    // Disabled to prevent European language false positives (Scandinavian, French, Portuguese, etc.)
    return false;
  }

  containsUrdu(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const urduRegex = /[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFF]/;
    return urduRegex.test(text);
  }

  containsIndonesian(text) {
    if (!text) return false;
    if (this.isOnlyCommonPunctuation(text)) return false;
    const indonesianWords = ['saya', 'anda', 'mereka', 'kami', 'kita', 'yang', 'dengan', 'untuk', 'dari', 'pada', 'oleh', 'dari', 'atau', 'tetapi', 'jika', 'ketika', 'sebelum', 'sesudah', 'karena', 'jadi', 'maka', 'namun', 'lagi', 'sudah', 'akan', 'bisa', 'harus', 'mau', 'ingin', 'perlu', 'boleh', 'tidak', 'tidak', 'benar', 'salah', 'baik', 'buruk', 'besar', 'kecil', 'tinggi', 'rendah', 'panjang', 'pendek', 'baru', 'muda', 'tua', 'cantik', 'ganteng', 'baik', 'jahat', 'senang', 'sedih', 'marah', 'takut', 'heran', 'bangga', 'malu', 'sakit', 'sehat', 'miskin', 'miskin', 'miskin'];
    const lowerText = text.toLowerCase();
    return indonesianWords.some(word => {
      const regex = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${word}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      return regex.test(lowerText);
    });
  }

  containsMalay(text) {
    // Commented out to prevent false positives with common English words
    return false;
    /*
    if (!text) return false;
    if (this.isOnlyCommonPunctuation(text)) return false;
    const malayWords = ['saya', 'awak', 'mereka', 'kami', 'kita', 'yang', 'dengan', 'untuk', 'dari', 'pada', 'oleh', 'dari', 'atau', 'tetapi', 'jika', 'ketika', 'sebelum', 'sesudah', 'karena', 'jadi', 'maka', 'namun', 'lagi', 'sudah', 'akan', 'bisa', 'harus', 'mau', 'ingin', 'perlu', 'boleh', 'tidak', 'tidak', 'benar', 'salah', 'baik', 'buruk', 'besar', 'kecil', 'tinggi', 'rendah', 'panjang', 'pendek', 'baru', 'muda', 'tua', 'cantik', 'ganteng', 'baik', 'jahat', 'senang', 'sedih', 'marah', 'takut', 'heran', 'bangga', 'malu', 'sakit', 'sehat', 'miskin'];
    const lowerText = text.toLowerCase();
    return malayWords.some(word => {
      const regex = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${word}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      return regex.test(lowerText);
    });
    */
  }

  containsBengali(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const bengaliRegex = /[\u0980-\u09FF]/;
    return bengaliRegex.test(text);
  }

  containsSinhala(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const sinhalaRegex = /[\u0D80-\u0DFF]/;
    return sinhalaRegex.test(text);
  }

  containsGurmukhi(text) {
    if (!text) return false;
    // Skip if text only contains common punctuation
    if (this.isOnlyCommonPunctuation(text)) return false;
    const gurmukhiRegex = /[\u0A00-\u0A7F]/;
    return gurmukhiRegex.test(text);
  }

  containsPashto(text) {
    if (!text) return false;
    const pashtoRegex = /[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFF]/;
    return pashtoRegex.test(text);
  }

  containsDari(text) {
    if (!text) return false;
    const dariRegex = /[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFF]/;
    return dariRegex.test(text);
  }

  containsAzerbaijani(text) {
    // Disabled to prevent European language false positives (French, Italian, Spanish, etc.)
    return false;
  }

  containsTajik(text) {
    // Disabled to prevent Cyrillic false positives (Russian, Ukrainian, Bulgarian, etc.)
    return false;
  }

  containsUzbek(text) {
    // Disabled to prevent European language false positives (same characters as Turkish)
    return false;
  }

  containsHebrew(text) {
    if (!text) return false;
    const hebrewRegex = /[\u0590-\u05FF]/;
    return hebrewRegex.test(text);
  }

  // Safely escape user-provided strings for regex construction
  escapeRegex(text) {
    return text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
  }

  containsIndianName(text) {
    if (!text) return false;
    const lowerText = text.toLowerCase();
    const matchedNames = [];
    const result = this.indianNames.some(name => {
      const nameLower = String(name).toLowerCase();
      const escaped = this.escapeRegex(nameLower);
      // Strict whole-word first
      const strict = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      // Relaxed username boundary: allow non-letter/digit/underscore/latin-accent on edges
      const relaxed = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      if (strict.test(lowerText) || relaxed.test(lowerText)) {
        matchedNames.push(name);
        return true;
      }
      return false;
    });
    if (result) {
      console.log('JeetBlock: Indian name match:', text, 'matched names:', matchedNames);
    }
    return result;
  }

  containsMuslimName(text) {
    if (!text) return false;
    const lowerText = text.toLowerCase();
    return this.muslimNames.some(name => {
      const nameLower = String(name).toLowerCase();
      const escaped = this.escapeRegex(nameLower);
      const strict = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      const relaxed = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      return strict.test(lowerText) || relaxed.test(lowerText);
    });
  }

  containsJewishName(text) {
    if (!text) return false;
    const lowerText = text.toLowerCase();
    const matchedNames = [];
    const result = this.jewishNames.some(name => {
      const nameLower = String(name).toLowerCase();
      const escaped = this.escapeRegex(nameLower);
      const strict = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      const relaxed = new RegExp(`(^|[^A-Za-z0-9_À-ÖØ-öø-ÿ])${escaped}([^A-Za-z0-9_À-ÖØ-öø-ÿ]|$)`, 'i');
      if (strict.test(lowerText) || relaxed.test(lowerText)) {
        matchedNames.push(name);
        return true;
      }
      return false;
    });
    if (result) {
      console.log('JeetBlock: ✓ Jewish name MATCH:', matchedNames.join(', '), 'in:', text);
    }
    return result;
  }

  containsIndianEmoji(text) {
    if (!text) return false;
    const result = Object.values(this.indianEmojis).some(emojis =>
      emojis.some(emoji => {
        // Use word boundary matching to ensure we match complete emojis
        const regex = new RegExp(`\\b${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\b|^${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}|${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`);
        return regex.test(text);
      })
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsIndianEmoji matched:', text);
    }
    return result;
  }

    containsMuslimEmoji(text) {
    if (!text) return false;
    
    // Special exclusion for German flag to prevent false positive with Sudan flag
    if (text.includes('🇩🇪')) {
      console.log('JeetBlock: DEBUG - German flag detected, excluding from Muslim emoji check');
      return false;
    }
    
    const result = Object.values(this.muslimEmojis).some(emojis =>
      emojis.some(emoji => {
        // Use word boundary matching to ensure we match complete emojis
        const regex = new RegExp(`\\b${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\b|^${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}|${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`);
        return regex.test(text);
      })
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsMuslimEmoji matched:', text);
    }
    return result;
  }

  containsJewishEmoji(text) {
    if (!text) return false;
    const result = Object.values(this.jewishEmojis).some(emojis =>
      emojis.some(emoji => {
        // Use word boundary matching to ensure we match complete emojis
        const regex = new RegExp(`\\b${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\b|^${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}|${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`);
        return regex.test(text);
      })
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsJewishEmoji matched:', text);
    }
    return result;
  }

  containsLGBTQIAEmoji(text) {
    if (!text) return false;
    const result = Object.values(this.lgbtqiaEmojis).some(emojis =>
      emojis.some(emoji => text.includes(emoji))
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsLGBTQIAEmoji matched:', text);
    }
    return result;
  }

  containsUkraineEmoji(text) {
    if (!text) return false;
    const result = Object.values(this.ukraineEmojis).some(emojis =>
      emojis.some(emoji => {
        // Use word boundary matching to ensure we match complete emojis
        const regex = new RegExp(`\\b${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\b|^${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}|${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`);
        return regex.test(text);
      })
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsUkraineEmoji matched:', text);
    }
    return result;
  }

  containsPalestineEmoji(text) {
    if (!text) return false;
    const result = Object.values(this.palestineEmojis).some(emojis =>
      emojis.some(emoji => {
        // Use word boundary matching to ensure we match complete emojis
        const regex = new RegExp(`\\b${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\b|^${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}|${emoji.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`);
        return regex.test(text);
      })
    );
    if (result) {
      console.log('JeetBlock: DEBUG - containsPalestineEmoji matched:', text);
    }
    return result;
  }

  containsCustomFilter(text) {
    if (!text || this.customFilters.length === 0) return false;
    const lowerText = text.toLowerCase();
    return this.customFilters.some(filter => 
      lowerText.includes(filter.toLowerCase())
    );
  }

  shouldFilterPost(post) {
    // Security check - if compromised, don't filter
    if (window.jeetBlockSecurity && window.jeetBlockSecurity.isExtensionCompromised()) {
      return false;
    }
    
    // Check if extension is enabled
    if (!this.isEnabled) {
      console.log('JeetBlock: Extension is disabled, not filtering post');
      return false;
    }
    
    // Never filter search input elements or navigation elements
    if (post.matches && (
      post.matches('input[type="search"]') ||
      post.matches('input[placeholder*="search"]') ||
      post.matches('input[aria-label*="search"]') ||
      post.matches('[role="searchbox"]') ||
      post.matches('[data-testid*="search"]') ||
      post.matches('form[role="search"]') ||
      post.closest('form[role="search"]') ||
      post.closest('[data-testid*="search"]')
    )) {
      console.log('JeetBlock: Skipping search element to preserve search functionality');
      return false;
    }
    
    // Skip LinkedIn posts that should be handled by specific observers
    const platform = this.getPlatform();
    if (platform === 'linkedin') {
      if (post.matches && (
        post.matches('article.main-feed-activity-card') ||
        post.matches('article.feed-reshare-content') ||
        post.matches('section.comment') ||
        post.matches('li a[href*="/in/"]')
      )) {
        console.log('JeetBlock: Skipping LinkedIn post - handled by specific observer');
        return false;
      }
    }
    
    // Skip filtering for "Suggested for you" posts (except on LinkedIn where we want to filter them)
    if (platform !== 'linkedin') {
      const postText = post.textContent || post.innerText || '';
      if (postText.toLowerCase().includes('suggested for you') || 
          postText.toLowerCase().includes('suggested')) {
        console.log('JeetBlock: Skipping "Suggested for you" post');
        return false;
      }
    }

    // Check if filtering is enabled for this platform
    let platformFilterKey;
    if (platform === 'linkedin') {
      platformFilterKey = 'filterLinkedIn';
    } else if (platform === 'reddit') {
      platformFilterKey = 'filterReddit';
    } else if (platform === 'tiktok') {
      platformFilterKey = 'filterTikTok';
    } else if (platform === 'youtube') {
      platformFilterKey = 'filterYouTube';
    } else {
      platformFilterKey = `filter${platform.charAt(0).toUpperCase() + platform.slice(1)}`;
    }
    
    console.log('JeetBlock: Platform:', platform, 'Filter key:', platformFilterKey, 'Value:', this[platformFilterKey]);
    
    if (!this[platformFilterKey]) {
      console.log('JeetBlock: Filtering disabled for platform:', platform, 'key:', platformFilterKey, 'value:', this[platformFilterKey]);
      return false; // Filtering disabled for this platform
    }

    // Check for quote tweets first - if this is a Twitter quote tweet, handle it specially
    if (platform === 'twitter') {
      try {
        console.log('JeetBlock: DEBUG - shouldFilterPost checking for quote tweets');
        const quotedCandidates = Array.from(post.querySelectorAll('div[role="link"]'))
          .filter(el => el.querySelector('[data-testid="Tweet-User-Avatar"]') || el.querySelector('[data-testid="User-Name"]'));

        console.log('JeetBlock: DEBUG - shouldFilterPost found quoted candidates:', quotedCandidates.length);
        
        if (quotedCandidates.length) {
          console.log('JeetBlock: DEBUG - This is a quote tweet, checking only quoted content');
          // For quote tweets, check if the quoted content should be filtered
          // but don't filter the entire post based on the outer author
          
          // Check if any quoted content should be filtered
          for (const quoted of quotedCandidates) {
            let quotedAuthor = '';
            const qName = quoted.querySelector('[data-testid="User-Name"] span, [data-testid="User-Name"] a');
            if (qName) quotedAuthor = this.getTextWithEmojis(qName);
            const quotedTextEl = quoted.querySelector('[data-testid="tweetText"]') || quoted.querySelector('[lang]');
            let quotedText = '';
            if (quotedTextEl) quotedText = this.getTextWithEmojis(quotedTextEl);
            const qCombined = (quotedAuthor + ' ' + quotedText).trim();
            const qAuthorLower = (quotedAuthor || '').toLowerCase();
            
            let quotedShouldFilter = false;
            if (this.containsJewishName(qAuthorLower) || this.containsMuslimName(qAuthorLower) || this.containsIndianName(qAuthorLower)) {
              quotedShouldFilter = true;
            }
            if (!quotedShouldFilter) {
              if (
                this.containsHebrew(qCombined) ||
                this.containsArabic(qCombined) || this.containsPersian(qCombined) || this.containsUrdu(qCombined) ||
                this.containsDevanagari(qCombined) || this.containsBengali(qCombined) || this.containsSinhala(qCombined) || this.containsGurmukhi(qCombined) ||
                this.containsLGBTQIAEmoji(qCombined) || this.containsUkraineEmoji(qCombined) || this.containsPalestineEmoji(qCombined) ||
                this.containsIndianEmoji(qCombined) || this.containsMuslimEmoji(qCombined) || this.containsJewishEmoji(qCombined)
              ) {
                quotedShouldFilter = true;
              }
            }
            
            if (quotedShouldFilter) {
              console.log('JeetBlock: DEBUG - Quoted content should be filtered:', quotedAuthor);
              return true; // Return true so applyFilter gets called to handle the filtering
            }
          }
          
          // If no quoted content should be filtered, don't filter the entire post
          console.log('JeetBlock: DEBUG - No quoted content should be filtered, not filtering entire post');
          return false;
        }
      } catch (_) {}
    }

    // Twitter DM-specific handling: filter DM rows and conversation containers
    if (platform === 'twitter') {
      // DM list row
      if (post.matches && post.matches('div[role="link"][data-testid="conversation"]')) {
        let dmName = '';
        const s1 = post.querySelector('div[dir="ltr"] span');
        if (s1) dmName = this.getTextWithEmojis(s1);
        if (!dmName) {
          const s2 = post.querySelector('[dir="auto"] span, [role="link"] [dir="ltr"] span');
          if (s2) dmName = this.getTextWithEmojis(s2);
        }
        const nm = (dmName || '').toLowerCase();
        if (nm && (this.containsJewishName(nm) || this.containsMuslimName(nm) || this.containsIndianName(nm))) {
          return true;
        }
      }
      // Main DM conversation container
      if (post.matches && post.matches('div[data-testid="DmActivityContainer"]')) {
        let header = '';
        const h = post.querySelector('h2[aria-level="2"] span span');
        if (h) header = this.getTextWithEmojis(h);
        const nm = (header || '').toLowerCase();
        if (nm && (this.containsJewishName(nm) || this.containsMuslimName(nm) || this.containsIndianName(nm))) {
          return true;
        }
      }
      // Conversation details pane
      if (post.matches && post.matches('section[aria-label="Section details"]')) {
        let header = '';
        const h = post.querySelector('#detail-header span span');
        if (h && h.textContent) header = h.textContent.trim();
        const nm = (header || '').toLowerCase();
        if (nm && (this.containsJewishName(nm) || this.containsMuslimName(nm) || this.containsIndianName(nm))) {
          return true;
        }
      }
    }

    // Facebook-specific username extraction
    let usernameElements = post.querySelectorAll(this.getUsernameSelectors());
    
    // For Facebook, also try to extract usernames from aria-label attributes
    if (platform === 'facebook') {
      const facebookUsernameElements = post.querySelectorAll('a[aria-label*=" "], svg[aria-label*=" "], [aria-label*="· Original audio"], [aria-label*="·"]');
      usernameElements = [...usernameElements, ...facebookUsernameElements];
    }
    
    // For Instagram, also try to extract usernames from profile picture alt text and href attributes
    if (platform === 'instagram') {
      const instagramUsernameElements = post.querySelectorAll('img[alt*="profile picture"], a[href*="/"][role="link"]:not([href*="/p/"]):not([href*="/reels/"]):not([href*="/explore/"]):not([href*="/stories/"])');
      usernameElements = [...usernameElements, ...instagramUsernameElements];
      console.log('JeetBlock: Instagram username elements found:', instagramUsernameElements.length);
      
      // Debug: Log all found elements
      instagramUsernameElements.forEach((el, index) => {
        console.log(`JeetBlock: Instagram element ${index}:`, {
          tagName: el.tagName,
          className: el.className,
          href: el.href,
          alt: el.alt,
          textContent: el.textContent?.trim(),
          innerText: el.innerText?.trim()
        });
      });
    }
    
    // For LinkedIn, also try to extract usernames from aria-label attributes
    if (platform === 'linkedin') {
      const linkedinUsernameElements = post.querySelectorAll('a[aria-label]');
      usernameElements = [...usernameElements, ...linkedinUsernameElements];
      console.log('JeetBlock: LinkedIn username elements found:', linkedinUsernameElements.length);
    }
    
    // For Reddit, also try to extract usernames from various Reddit-specific selectors
    if (platform === 'reddit') {
      console.log('JeetBlock: Processing Reddit post:', post);
      
      const redditUsernameElements = post.querySelectorAll('a[href*="/user/"], a[href*="/u/"], [slot="authorName"] a, .author, .username');
      usernameElements = [...usernameElements, ...redditUsernameElements];
      console.log('JeetBlock: Reddit username elements found:', redditUsernameElements.length);
      
      // Debug: Log all found username elements
      redditUsernameElements.forEach((el, index) => {
        console.log(`JeetBlock: Reddit username element ${index}:`, {
          tagName: el.tagName,
          className: el.className,
          href: el.href,
          textContent: el.textContent?.trim(),
          innerText: el.innerText?.trim()
        });
      });
      
      // Also look for usernames in spans that contain "u/" pattern
      const redditSpans = post.querySelectorAll('span');
      console.log('JeetBlock: Checking', redditSpans.length, 'spans for u/ pattern');
      redditSpans.forEach((span, index) => {
        const text = this.getTextWithEmojis(span) || span.innerText;
        if (text && text.includes('u/') && text.match(/^u\/\w+$/)) {
          console.log(`JeetBlock: Found u/ pattern in span ${index}:`, text);
          usernameElements.push(span);
        }
      });
      
      console.log('JeetBlock: Total Reddit username elements after processing:', usernameElements.length);
    }
    
    // For TikTok, also try to extract usernames from TikTok-specific selectors
    if (platform === 'tiktok') {
      console.log('JeetBlock: Processing TikTok post:', post);
      
      const tiktokUsernameElements = post.querySelectorAll('a[href*="/@"], [data-e2e="video-author-uniqueid"], [data-e2e="video-author-avatar"], [data-e2e="explore-card-user-link"], [data-e2e="explore-card-user-unique-id"]');
      usernameElements = [...usernameElements, ...tiktokUsernameElements];
      console.log('JeetBlock: TikTok username elements found:', tiktokUsernameElements.length);
      
      // Debug: Log all found username elements
      tiktokUsernameElements.forEach((el, index) => {
        console.log(`JeetBlock: TikTok username element ${index}:`, {
          tagName: el.tagName,
          className: el.className,
          href: el.href,
          textContent: el.textContent?.trim(),
          innerText: el.innerText?.trim(),
          dataE2e: el.getAttribute('data-e2e')
        });
      });
      
      console.log('JeetBlock: Total TikTok username elements after processing:', usernameElements.length);
      
      // TikTok username filtering (normal mode)
      // Username filtering will be handled by the main filtering logic below

      // Fallback: Right-side fullscreen panel often sits outside the post container.
      // If we still have no username elements, probe the visible right-rail profile wrapper.
      if (usernameElements.length === 0) {
        const rightRail = document.querySelector('div[class*="--DivProfileWrapper"]');
        if (rightRail) {
          const panelEls = rightRail.querySelectorAll('a[href^="/@"], [data-e2e="browse-username"], [data-e2e="browser-nickname"]');
          console.log('JeetBlock: TikTok right-rail username elements found:', panelEls.length);
          usernameElements = [...usernameElements, ...panelEls];
        }
      }
    }
    
    // For YouTube, also try to extract usernames from YouTube-specific selectors
    if (platform === 'youtube') {
      console.log('JeetBlock: Processing YouTube comment:', post);
      
      const youtubeUsernameElements = post.querySelectorAll('#author-text');
      usernameElements = [...usernameElements, ...youtubeUsernameElements];
      console.log('JeetBlock: YouTube username elements found:', youtubeUsernameElements.length);
      
      // Debug: Log all found username elements
      youtubeUsernameElements.forEach((el, index) => {
        console.log(`JeetBlock: YouTube username element ${index}:`, {
          tagName: el.tagName,
          className: el.className,
          href: el.href,
          textContent: el.textContent?.trim(),
          innerText: el.innerText?.trim()
        });
      });
      
      console.log('JeetBlock: Total YouTube username elements after processing:', usernameElements.length);
    }

    // Check allow/block lists first
    for (const element of usernameElements) {
      let username = element.textContent || element.innerText;
      
      // For Facebook, also check aria-label for usernames
      if (platform === 'facebook' && element.getAttribute('aria-label')) {
        const ariaLabel = element.getAttribute('aria-label');
        // Extract name from aria-label (e.g., "Mohammed Attia" from "Mohammed Attia · Original audio")
        if (ariaLabel && ariaLabel !== username) {
          // Split by common separators and take the first part (the name)
          const namePart = ariaLabel.split('·')[0].split('•')[0].split('|')[0].trim();
          if (namePart && namePart.length > 0) {
            username = namePart;
          } else {
            username = ariaLabel;
          }
        }
      }
      
      // For Instagram, also check alt text for usernames
      if (platform === 'instagram' && element.getAttribute('alt')) {
        const altText = element.getAttribute('alt');
        if (altText && altText.includes("profile picture")) {
          // Extract username from alt text (e.g., "womanpropaganda's profile picture")
          const usernameMatch = altText.match(/([^']+)'s profile picture/);
          if (usernameMatch && usernameMatch[1]) {
            username = usernameMatch[1];
            console.log('JeetBlock: Instagram username from alt text:', username);
          }
        }
      }
      
      // For Instagram, also check href for usernames
      if (platform === 'instagram' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && href.startsWith('/') && !href.includes('/p/') && !href.includes('/reels/')) {
          // Extract username from href (e.g., "/womanpropaganda/")
          const usernameMatch = href.match(/\/([^\/]+)\//);
          if (usernameMatch && usernameMatch[1]) {
            username = usernameMatch[1];
            console.log('JeetBlock: Instagram username from href:', username);
          }
        }
      }
      
      // For LinkedIn, also check aria-label for usernames
      if (platform === 'linkedin' && element.getAttribute('aria-label')) {
        const ariaLabel = element.getAttribute('aria-label');
        const visibleName = this.getTextWithEmojis(element) || element.innerText;
        
        // Use both aria-label and visible text for username detection
        if (ariaLabel && ariaLabel.trim()) {
          username = ariaLabel.trim();
          console.log('JeetBlock: LinkedIn username from aria-label:', username);
        } else if (visibleName && visibleName.trim()) {
          username = visibleName.trim();
          console.log('JeetBlock: LinkedIn username from text content:', username);
        }
      }
      
      // For Reddit, also check href for usernames
      if (platform === 'reddit' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && (href.includes('/user/') || href.includes('/u/'))) {
          // Extract username from href (e.g., "/user/username" or "/u/username")
          const usernameMatch = href.match(/\/(?:user|u)\/([^\/]+)/);
          if (usernameMatch && usernameMatch[1]) {
            username = usernameMatch[1];
            console.log('JeetBlock: Reddit username from href:', username);
          }
        }
      }
      
      // For TikTok, also check href for usernames
      if (platform === 'tiktok' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && href.includes('/@')) {
          // Extract username from href (e.g., "/@username")
          const usernameMatch = href.match(/\/@([^\/]+)/);
          if (usernameMatch && usernameMatch[1]) {
            username = usernameMatch[1];
            console.log('JeetBlock: TikTok username from href:', username);
          }
        }
      }
      
      // For TikTok explore page, check data-e2e attribute for usernames
      if (platform === 'tiktok' && element.getAttribute('data-e2e') === 'explore-card-user-unique-id') {
        const exploreUsername = this.getTextWithEmojis(element) || element.innerText;
        if (exploreUsername && exploreUsername.trim()) {
          username = exploreUsername.trim();
          console.log('JeetBlock: TikTok explore username from data-e2e:', username);
        }
      }
      
      // For YouTube, extract username from href and clean up @ symbol
      if (platform === 'youtube' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && href.includes('/@')) {
          // Extract username from href (e.g., "/@kathopiai")
          const usernameMatch = href.match(/\/@([^\/]+)/);
          if (usernameMatch && usernameMatch[1]) {
            username = usernameMatch[1];
            console.log('JeetBlock: YouTube username from href:', username);
          }
        }
      }
      
      // For YouTube, also clean up @ symbol from text content
      if (platform === 'youtube' && username && username.includes('@')) {
        username = username.replace('@', '').trim();
        console.log('JeetBlock: YouTube username cleaned:', username);
      }
      
      // For Reddit, also check for "u/username" pattern in text content
      if (platform === 'reddit' && username && username.includes('u/')) {
        console.log('JeetBlock: Found u/ pattern in username:', username);
        const usernameMatch = username.match(/u\/(\w+)/);
        if (usernameMatch && usernameMatch[1]) {
          username = usernameMatch[1];
          console.log('JeetBlock: Reddit username from u/ pattern:', username);
        }
      }
      
      // Debug logging for Reddit username processing
      if (platform === 'reddit') {
        console.log('JeetBlock: Processing Reddit username:', username, 'from element:', element);
      }
      
      const usernameLower = username.toLowerCase();
      
      // Skip checking non-username elements for Instagram
      if (platform === 'instagram' && username && username.trim()) {
        // Skip timestamps, like counts, hashtags, and other non-username text
        if (username.match(/^\d+[hm]$/) || // timestamps like "21h", "11h"
            username.match(/^\d+,\d+ likes?$/) || // like counts like "2,527 likes"
            username.match(/^#\w+$/) || // hashtags like "#usa"
            username.match(/^View all \d+ comments?$/) || // comment counts
            username.match(/^Original audio$/) || // audio labels
            username.length > 50 || // very long text (likely post content)
            username.includes('...') || // truncated text
            username.includes('more')) { // "more" text
          return false; // Skip this element
        }
        console.log('JeetBlock: Checking Instagram username:', username, 'against blocklist');
      }
      
      // Check if username is in allow list (exempt from filtering)
      if (this.allowList && this.allowList.some(allowed => 
        usernameLower.includes(allowed.toLowerCase())
      )) {
        console.log('JeetBlock: Username in allow list, not filtering:', username);
        return false;
      }
      
      // Check if username is in block list (force filtering)
      if (this.blockList && this.blockList.some(blocked => 
        usernameLower.includes(blocked.toLowerCase())
      )) {
        console.log('JeetBlock: Username in block list, forcing filter:', username);
        return true;
      }
      
      // Debug: Log what username is being checked
      if (platform === 'reddit') {
        console.log('JeetBlock: Checking Reddit username against blocklists:', username);
      }
      
      // Debug: Log what's being checked for "Suggested for you" posts
      if (platform === 'instagram' && username && username.toLowerCase().includes('suggested')) {
        console.log('JeetBlock: DEBUG - Suggested post username detected:', username);
      }
    }

    const contentElements = post.querySelectorAll(this.getContentSelectors());

    // Check usernames (STRICT - check everything: scripts, names, emojis)
    for (const element of usernameElements) {
      let text = this.getTextWithEmojis(element) || element.innerText;
      
      // For Facebook, also check aria-label for usernames
      if (platform === 'facebook' && element.getAttribute('aria-label')) {
        const ariaLabel = element.getAttribute('aria-label');
        if (ariaLabel && ariaLabel !== text) {
          // Extract name from aria-label (e.g., "Mohammed Attia" from "Mohammed Attia · Original audio")
          const namePart = ariaLabel.split('·')[0].split('•')[0].split('|')[0].trim();
          if (namePart && namePart.length > 0) {
            text = namePart;
          } else {
            text = ariaLabel;
          }
        }
      }
      
      // For Instagram, also check alt text for usernames
      if (platform === 'instagram' && element.getAttribute('alt')) {
        const altText = element.getAttribute('alt');
        if (altText && altText.includes("profile picture")) {
          // Extract username from alt text (e.g., "womanpropaganda's profile picture")
          const usernameMatch = altText.match(/([^']+)'s profile picture/);
          if (usernameMatch && usernameMatch[1]) {
            text = usernameMatch[1];
          }
        }
      }
      
      // For YouTube, also check href for usernames
      if (platform === 'youtube' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && href.includes('/@')) {
          // Extract username from href (e.g., "/@kathopiai")
          const usernameMatch = href.match(/\/@([^\/]+)/);
          if (usernameMatch && usernameMatch[1]) {
            text = usernameMatch[1];
          }
        }
      }
      
      // For Instagram, also check href for usernames
      if (platform === 'instagram' && element.getAttribute('href')) {
        const href = element.getAttribute('href');
        if (href && href.startsWith('/') && !href.includes('/p/') && !href.includes('/reels/')) {
          // Extract username from href (e.g., "/womanpropaganda/")
          const usernameMatch = href.match(/\/([^\/]+)\//);
          if (usernameMatch && usernameMatch[1]) {
            text = usernameMatch[1];
          }
        }
      }
      
      const textLower = text.toLowerCase();
      
      // Debug: Log what username is being checked for ALL platforms
      console.log(`JeetBlock: [${platform.toUpperCase()}] Checking username for filtering:`, text);
      
      // Indian filtering
      if (this.filterIndian) {
        const reasons = [];
        if (this.containsDevanagari(text)) reasons.push('Devanagari script');
        if (this.containsBengali(text)) reasons.push('Bengali script');
        if (this.containsSinhala(text)) reasons.push('Sinhala script');
        if (this.containsGurmukhi(text)) reasons.push('Gurmukhi script');
        if (this.containsIndianName(textLower)) reasons.push('Indian name');
        if (this.containsIndianEmoji(text)) reasons.push('Indian emoji');
        
        if (reasons.length > 0) {
          console.log('JeetBlock: Filtering Indian username:', text, 'Reasons:', reasons.join(', '));
          return true;
        }
      }
      
      // Debug: Log what's being filtered for "Suggested for you" posts
      if (platform === 'instagram' && text && text.toLowerCase().includes('suggested')) {
        console.log('JeetBlock: DEBUG - Suggested post content detected:', text);
      }
      
      // Muslim/Pakistani filtering
      if (this.filterMuslim) {
        const reasons = [];
        if (this.containsArabic(text)) reasons.push('Arabic script');
        if (this.containsPersian(text)) reasons.push('Persian script');
        // if (this.containsTurkish(text)) reasons.push('Turkish characters'); // Disabled to prevent false positives
        if (this.containsUrdu(text)) reasons.push('Urdu script');
        if (this.containsIndonesian(text)) reasons.push('Indonesian words');
        if (this.containsMalay(text)) reasons.push('Malay words');
        if (this.containsPashto(text)) reasons.push('Pashto script');
        if (this.containsDari(text)) reasons.push('Dari script');
        // if (this.containsAzerbaijani(text)) reasons.push('Azerbaijani script'); // Disabled to prevent European language false positives
        // if (this.containsTajik(text)) reasons.push('Tajik script'); // Disabled to prevent Cyrillic false positives
        // if (this.containsUzbek(text)) reasons.push('Uzbek script'); // Disabled to prevent false positives
        if (this.containsMuslimName(textLower)) reasons.push('Muslim name');
        if (this.containsMuslimEmoji(text)) reasons.push('Muslim emoji');
        
        // Debug logging for Reddit
        if (platform === 'reddit') {
          console.log('JeetBlock: Checking Muslim filter for Reddit username:', text, 'Lowercase:', textLower);
          console.log('JeetBlock: Muslim name check result:', this.containsMuslimName(textLower));
        }
        
        if (reasons.length > 0) {
          console.log('JeetBlock: Filtering Muslim username:', text, 'Reasons:', reasons.join(', '));
          return true;
        }
      }
      
      // Jewish/Israeli filtering
      if (this.filterJewish) {
        const reasons = [];
        if (this.containsHebrew(text)) reasons.push('Hebrew script');
        if (this.containsJewishName(textLower)) reasons.push('Jewish name');
        if (this.containsJewishEmoji(text)) reasons.push('Jewish emoji');
        
        if (reasons.length > 0) {
          console.log('JeetBlock: Filtering Jewish username:', text, 'Reasons:', reasons.join(', '));
          return true;
        }
      }
      
      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(text)) {
        console.log('JeetBlock: Filtering LGBTQIA+ username:', text, 'Reason: LGBTQIA+ emoji');
        return true;
      }
      
      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(text)) {
        console.log('JeetBlock: Filtering Ukraine username:', text, 'Reason: Ukraine emoji');
        return true;
      }
      
      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(text)) {
        console.log('JeetBlock: Filtering Palestine username:', text, 'Reason: Palestine emoji');
        return true;
      }
      
      // Universal filters (always active)
      if (this.containsCustomFilter(textLower)) {
        console.log('JeetBlock: Filtering custom filter username:', text, 'Reason: Custom filter');
        return true;
      }
    }

    // Check content (SELECTIVE - only scripts and emojis, NO names)
    for (const element of contentElements) {
      const text = element.textContent || element.innerText;
      
      // Indian filtering (scripts and emojis only)
      if (this.filterIndian && (
        this.containsDevanagari(text) || 
        this.containsBengali(text) || 
        this.containsSinhala(text) || 
        this.containsGurmukhi(text) || 
        this.containsIndianEmoji(text)
      )) {
        console.log('JeetBlock: Filtering Indian content:', text.substring(0, 100));
        return true;
      }
      
      // Muslim/Pakistani filtering (scripts and emojis only)
      if (this.filterMuslim && (
        this.containsArabic(text) || 
        this.containsPersian(text) ||
        // this.containsTurkish(text) || // Disabled to prevent false positives
        this.containsUrdu(text) ||
        this.containsIndonesian(text) ||
        this.containsMalay(text) ||
        this.containsPashto(text) ||
        this.containsDari(text) ||
        // this.containsAzerbaijani(text) || // Disabled to prevent European language false positives
        // this.containsTajik(text) || // Disabled to prevent Cyrillic false positives
        // this.containsUzbek(text) || // Disabled to prevent false positives
        this.containsMuslimEmoji(text)
      )) {
        return true;
      }
      
      // Jewish/Israeli filtering (scripts and emojis only)
      if (this.filterJewish) {
        const reasons = [];
        if (this.containsHebrew(text)) reasons.push('Hebrew script');
        if (this.containsJewishEmoji(text)) reasons.push('Jewish emoji');
        
        if (reasons.length > 0) {
          console.log('JeetBlock: Filtering Jewish content:', text, 'Reasons:', reasons.join(', '));
          return true;
        }
      }
      
      // LGBTQIA+ filtering
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(text)) {
        return true;
      }
      
      // Ukraine flag filtering
      if (this.filterUkraine && this.containsUkraineEmoji(text)) {
        return true;
      }
      
      // Palestine flag filtering
      if (this.filterPalestine && this.containsPalestineEmoji(text)) {
        return true;
      }
      
      // Universal filters (always active)
      if (this.containsCustomFilter(text.toLowerCase())) {
        return true;
      }
    }

    return false;
  }

  applyFilter(post, type = null) {
    // Prevent duplicate processing - if already filtered, don't process again
    if (post.dataset.harmonizerFiltered === 'collapsed' || post.dataset.harmonizerFiltered === 'removed') {
      console.log('JeetBlock: Post already filtered, skipping duplicate processing');
      return;
    }
    
    // Debug: Log if this is a YouTube video element
    if (post.tagName && post.tagName.toLowerCase().includes('ytd-rich-item')) {
      console.log('JeetBlock: applyFilter called on ytd-rich-item-renderer. Flags:', {
        harmonizerFiltered: post.dataset.harmonizerFiltered,
        harmonizerProcessed: post.dataset.harmonizerProcessed,
        ytVideo: post.dataset.ytVideo
      });
    }
    
    // Special handling for YouTube sidebar items
    if (type === 'youtube-sidebar') {
      this.applyYouTubeSidebarFilter(post);
      return;
    }
    
    // Special handling for TikTok
    if (this.platform === 'tiktok') {
      this.applyTikTokFilter(post);
      return;
    }

    // Special handling for Twitter DMs: these need special treatment but should still respect filter mode
    if (this.platform === 'twitter') {
      const isDmRow = post.matches && post.matches('div[role="link"][data-testid="conversation"]');
      const isDmContainer = post.matches && post.matches('div[data-testid="DmActivityContainer"]');
      const isDetailsPane = post.matches && post.matches('section[aria-label="Section details"]');
      if (isDmRow || isDmContainer || isDetailsPane) {
        // Only force-remove if user has set remove mode
        if (this.filterMode === 'remove') {
          post.style.display = 'none';
          post.dataset.harmonizerFiltered = 'removed';
          post.dataset.twitter = 'true';
          return;
        }
        // Otherwise fall through to normal collapse handling below
      }


      // Special handling for Twitter quoted/retweet posts.
      // Outer author should be filtered immediately if they match any criteria; otherwise collapse only the quoted block.
      try {
        console.log('JeetBlock: DEBUG - Starting quote tweet detection for post:', post);
        const quotedCandidates = Array.from(post.querySelectorAll('div[role="link"]'))
          .filter(el => el.querySelector('[data-testid="Tweet-User-Avatar"]') || el.querySelector('[data-testid="User-Name"]'));

        console.log('JeetBlock: DEBUG - Found quoted candidates:', quotedCandidates.length);
        console.log('JeetBlock: DEBUG - All div[role="link"] elements:', Array.from(post.querySelectorAll('div[role="link"]')).length);
        console.log('JeetBlock: DEBUG - All User-Name elements:', Array.from(post.querySelectorAll('[data-testid="User-Name"]')).length);

        if (quotedCandidates.length) {
          console.log('JeetBlock: DEBUG - applyFilter handling quote tweet, only filtering quoted content');
          
          // For quote tweets, only filter the quoted content, not the entire post
          for (const quoted of quotedCandidates) {
            let quotedAuthor = '';
            const qName = quoted.querySelector('[data-testid="User-Name"] span, [data-testid="User-Name"] a');
            if (qName) quotedAuthor = this.getTextWithEmojis(qName);
            const quotedTextEl = quoted.querySelector('[data-testid="tweetText"]') || quoted.querySelector('[lang]');
            let quotedText = '';
            if (quotedTextEl) quotedText = this.getTextWithEmojis(quotedTextEl);
            const qCombined = (quotedAuthor + ' ' + quotedText).trim();
            const qAuthorLower = (quotedAuthor || '').toLowerCase();
            
            let quotedShouldFilter = false;
            if (this.containsJewishName(qAuthorLower) || this.containsMuslimName(qAuthorLower) || this.containsIndianName(qAuthorLower)) {
              quotedShouldFilter = true;
            }
            if (!quotedShouldFilter) {
              if (
                this.containsHebrew(qCombined) ||
                this.containsArabic(qCombined) || this.containsPersian(qCombined) || this.containsUrdu(qCombined) ||
                this.containsDevanagari(qCombined) || this.containsBengali(qCombined) || this.containsSinhala(qCombined) || this.containsGurmukhi(qCombined) ||
                this.containsLGBTQIAEmoji(qCombined) || this.containsUkraineEmoji(qCombined) || this.containsPalestineEmoji(qCombined) ||
                this.containsIndianEmoji(qCombined) || this.containsMuslimEmoji(qCombined) || this.containsJewishEmoji(qCombined)
              ) {
                quotedShouldFilter = true;
              }
            }
            
            if (quotedShouldFilter) {
              console.log('JeetBlock: DEBUG - Collapsing only quoted content:', quotedAuthor);
              this.collapsePostElement(quoted);
            }
          }
          
          // Mark the post as processed but don't collapse the entire post
          post.dataset.harmonizerProcessed = 'true';
          post.dataset.twitter = 'true';
          return;
        }
      } catch (_) {}
    }

    this.collapsePostElement(post, type);
  }

  collapsePostElement(post, reason = null) {
    if (!post) return;
    if (reason) post.dataset.harmonizerReason = reason;
    
    if (this.filterMode === 'remove') {
      post.style.display = 'none';
      post.dataset.harmonizerFiltered = 'removed';
      return;
    }

    // Collapse/Hide mode - completely hide content and show message
    post.classList.add('harmonizer-collapsed');
    post.dataset.harmonizerFiltered = 'collapsed';

    // Add platform-specific data attribute for styling
    if (this.platform === 'facebook') {
      post.dataset.facebook = 'true';
    } else if (this.platform === 'instagram') {
      post.dataset.instagram = 'true';
    } else if (this.platform === 'linkedin') {
      post.dataset.linkedin = 'true';
    } else if (this.platform === 'reddit') {
      post.dataset.reddit = 'true';
    } else if (this.platform === 'tiktok') {
      post.dataset.tiktok = 'true';
    } else if (this.platform === 'twitter') {
      post.dataset.twitter = 'true';
    } else if (this.platform === 'youtube') {
      post.dataset.youtube = 'true';
    }

    // Store original content
    if (!post.dataset.originalContent) {
      post.dataset.originalContent = post.innerHTML;
    }

    // Create hidden message if it doesn't exist
    if (!post.querySelector('.harmonizer-hidden-message')) {
      const hiddenMsg = document.createElement('div');
      hiddenMsg.className = 'harmonizer-hidden-message';
      if (post.dataset.instagram === 'true' && post.dataset.igReply === 'true') {
        // Instagram replies: no toggle controls at all
        hiddenMsg.innerHTML = `
          <div class="harmonizer-hidden-text">This post has been hidden by JeetBlock</div>
        `;
      } else {
        hiddenMsg.innerHTML = `
          <div class="harmonizer-hidden-text">This post has been hidden by JeetBlock</div>
          <div class="harmonizer-toggle-btn" data-action="show">
            <span class="harmonizer-arrow">▼</span>
            <span class="harmonizer-toggle-text">Show</span>
          </div>
        `;
      }
      // Add click handler for toggle button
      // For Instagram replies, do not wire any toggle
      if (!(post.dataset.instagram === 'true' && post.dataset.igReply === 'true')) {
        const toggleBtn = hiddenMsg.querySelector('.harmonizer-toggle-btn');
        if (toggleBtn) {
          toggleBtn.onclick = (e) => {
            e.stopPropagation();
            this.togglePost(post);
          };
        }
      }
      if (post.dataset.reddit === 'true') {
        // For Reddit comments, only replace the comment body slot to preserve child replies
        const commentSlot = post.querySelector('[slot="comment"]');
        if (commentSlot) {
          if (!post.dataset.originalRedditComment) {
            post.dataset.originalRedditComment = commentSlot.innerHTML;
          }
          commentSlot.innerHTML = '';
          commentSlot.appendChild(hiddenMsg);
        } else {
          post.innerHTML = '';
          post.appendChild(hiddenMsg);
        }
      } else {
        // Replace whole content with hidden message
        post.innerHTML = '';
        post.appendChild(hiddenMsg);
      }
    } else {
      // Hidden message already exists, just ensure the click handler is properly set
      const existingToggleBtn = post.querySelector('.harmonizer-toggle-btn');
      if (existingToggleBtn && !existingToggleBtn.onclick) {
        existingToggleBtn.onclick = (e) => {
          e.stopPropagation();
          this.togglePost(post);
        };
      }
    }
  }

  applyTikTokFilter(post) {
    console.log('JeetBlock: Applying TikTok filter to post:', post);
    
    // Stop all videos in the post
    const videos = post.querySelectorAll('video');
    videos.forEach(video => {
      console.log('JeetBlock: Pausing TikTok video');
      video.pause();
      video.currentTime = 0;
      video.muted = true;
      video.volume = 0;
      video.style.pointerEvents = 'none';
    });
    
    // For TikTok explore page, also hide the username section
    if (post.getAttribute('data-e2e') === 'explore-item') {
      console.log('JeetBlock: TikTok explore item - looking for username section');
      const nextSibling = post.nextElementSibling;
      if (nextSibling && nextSibling.getAttribute('data-e2e') === 'explore-card-desc') {
        console.log('JeetBlock: Found username section, hiding it too');
        nextSibling.style.display = 'none';
        nextSibling.dataset.harmonizerFiltered = 'collapsed';
        // Store reference to username section for restoration
        post.dataset.usernameSection = nextSibling.outerHTML;
      }
    }
    
    // Store original content
    if (!post.dataset.originalContent) {
      post.dataset.originalContent = post.innerHTML;
    }
    
    // Hide the post content but keep the container
    post.style.maxHeight = '200px';
    post.style.overflow = 'hidden';
    // Ensure toggle anchor positioning is relative to the post
    if (!post.style.position) post.style.position = 'relative';
    post.classList.add('harmonizer-collapsed');
    post.dataset.harmonizerFiltered = 'collapsed';
    post.dataset.tiktok = 'true';
    
    // Create hidden message with Show button
    const hiddenMsg = document.createElement('div');
    hiddenMsg.className = 'harmonizer-hidden-message';
    hiddenMsg.innerHTML = `
      <div class="harmonizer-hidden-text">
        <strong>Content Filtered</strong><br>
        <span style="font-size: 12px; color: #666;">TikTok post hidden by JeetBlock</span>
      </div>
      <button class="harmonizer-toggle-btn harmonizer-reveal-btn" data-action="show" style="position: static; display: inline-flex; align-items: center; gap: 6px;">
        <span class="harmonizer-arrow">▼</span>
        <span class="harmonizer-toggle-text">Show</span>
      </button>
    `;
    
    // Add click handler for the show button
    const showBtn = hiddenMsg.querySelector('.harmonizer-toggle-btn');
    showBtn.onclick = (e) => {
      e.stopPropagation();
      this.togglePost(post);
    };
    
    // Replace post content with hidden message
    post.innerHTML = '';
    post.appendChild(hiddenMsg);
    
    console.log('JeetBlock: TikTok post filtered with Show button');
  }

  autoNextTikTokVideo(filteredPost) {
    // Try multiple methods to trigger next video
    
    // Method 1: Simulate down arrow key (most reliable)
    const downEvent = new KeyboardEvent('keydown', {
      key: 'ArrowDown',
      code: 'ArrowDown',
      keyCode: 40,
      which: 40,
      bubbles: true,
      cancelable: true
    });
    document.dispatchEvent(downEvent);
    
    // Method 2: Simulate spacebar (alternative navigation)
    setTimeout(() => {
      const spaceEvent = new KeyboardEvent('keydown', {
        key: ' ',
        code: 'Space',
        keyCode: 32,
        which: 32,
        bubbles: true,
        cancelable: true
      });
      document.dispatchEvent(spaceEvent);
    }, 100);
    
    // Method 3: Try to find and click TikTok's next button
    setTimeout(() => {
      const nextButtons = document.querySelectorAll('[data-e2e="arrow-right"], [data-e2e="browse-next"], .next-button, .arrow-right');
      if (nextButtons.length > 0) {
        nextButtons[0].click();
      }
    }, 200);
    
    // Method 4: Scroll down to trigger infinite scroll
    setTimeout(() => {
      const currentScroll = window.scrollY;
      window.scrollTo(0, currentScroll + 100);
    }, 300);
    
    // Method 5: Try to trigger TikTok's native next video function
    setTimeout(() => {
      // Look for TikTok's internal next video function
      if (window.tiktok && window.tiktok.nextVideo) {
        window.tiktok.nextVideo();
      }
      
      // Try to find and trigger any next video related functions
      const possibleFunctions = [
        'nextVideo',
        'skipVideo', 
        'next',
        'skip',
        'advance'
      ];
      
      possibleFunctions.forEach(funcName => {
        if (window[funcName] && typeof window[funcName] === 'function') {
          try {
            window[funcName]();
          } catch (e) {
            // Ignore errors
          }
        }
      });
    }, 400);
  }

  nextTikTokVideo(filteredPost) {
    // Find the next video in the feed
    const allVideos = Array.from(document.querySelectorAll(this.getPostSelectors()));
    const currentIndex = allVideos.indexOf(filteredPost);
    const nextVideo = allVideos[currentIndex + 1];
    
    if (nextVideo) {
      // Scroll to the next video
      nextVideo.scrollIntoView({ behavior: 'smooth', block: 'center' });
      
      // Optional: Trigger TikTok's next video mechanism
      // This simulates pressing the down arrow key
      const downEvent = new KeyboardEvent('keydown', {
        key: 'ArrowDown',
        code: 'ArrowDown',
        keyCode: 40,
        which: 40,
        bubbles: true
      });
      document.dispatchEvent(downEvent);
    } else {
      // If no next video, try to trigger TikTok's infinite scroll
      window.scrollTo(0, document.body.scrollHeight);
    }
  }

  togglePost(post) {
    console.log('JeetBlock: togglePost called');
    
    // Check if this is a hide button click
    const hideBtn = post.querySelector('.harmonizer-hide-btn');
    if (hideBtn) {
      console.log('JeetBlock: Hide button clicked, hiding post...');
      
      // For TikTok explore page, also hide the username section again
      if (post.getAttribute('data-e2e') === 'explore-item') {
        const nextSibling = post.nextElementSibling;
        if (nextSibling && nextSibling.getAttribute('data-e2e') === 'explore-card-desc') {
          console.log('JeetBlock: Hiding TikTok username section again');
          nextSibling.style.display = 'none';
          nextSibling.dataset.harmonizerFiltered = 'collapsed';
        }
      }
      
      // Clear the revealed flag and hide the post again
      delete post.dataset.harmonizerRevealed;
      this.applyFilter(post);
      return;
    }
    
    // Check if this is a show button click
    const toggleBtn = post.querySelector('.harmonizer-toggle-btn');
    if (toggleBtn && toggleBtn.dataset.action === 'show') {
      console.log('JeetBlock: Show button clicked, revealing post...');
      
      // Set flags BEFORE modifying DOM to prevent re-filtering
      post.dataset.harmonizerRevealed = 'true';
      post.dataset.harmonizerFiltered = 'revealed';
      console.log('JeetBlock: Flags set - harmonizerRevealed:', post.dataset.harmonizerRevealed, 'harmonizerFiltered:', post.dataset.harmonizerFiltered);
      
      // Remove the hidden message completely
      const hiddenMsg = post.querySelector('.harmonizer-hidden-message');
      if (hiddenMsg) {
        hiddenMsg.remove();
      }
      
      // Show the post by restoring original content
      post.innerHTML = post.dataset.originalContent;
      post.classList.remove('harmonizer-collapsed');
      
      // For TikTok explore page, also restore the username section
      if (post.getAttribute('data-e2e') === 'explore-item' && post.dataset.usernameSection) {
        console.log('JeetBlock: Restoring TikTok username section');
        const nextSibling = post.nextElementSibling;
        if (nextSibling && nextSibling.getAttribute('data-e2e') === 'explore-card-desc') {
          nextSibling.style.display = '';
          delete nextSibling.dataset.harmonizerFiltered;
        }
      }
      
      // For YouTube, restore video elements and thumbnails
      if (this.platform === 'youtube') {
        console.log('JeetBlock: Restoring YouTube content');
        // Restore video thumbnails
        const thumbnails = post.querySelectorAll('img[src*="ytimg.com"], img[src*="youtube.com"]');
        thumbnails.forEach(img => {
          img.style.display = '';
          img.style.visibility = 'visible';
          img.style.opacity = '1';
        });
        
        // Restore video containers
        const videoContainers = post.querySelectorAll('ytd-thumbnail, ytd-video-renderer, ytd-rich-item-renderer');
        videoContainers.forEach(container => {
          container.style.display = '';
          container.style.visibility = 'visible';
          container.style.opacity = '1';
        });
        
        // Restore any hidden video elements
        const videoElements = post.querySelectorAll('video, iframe[src*="youtube.com"]');
        videoElements.forEach(video => {
          video.style.display = '';
          video.style.visibility = 'visible';
          video.style.opacity = '1';
        });
      }
      
      // Re-set the flags after DOM modification to ensure they persist
      post.dataset.harmonizerRevealed = 'true';
      post.dataset.harmonizerFiltered = 'revealed';
      
      // Ensure the post is visible
      post.style.display = '';
      post.style.visibility = 'visible';
      post.style.opacity = '1';
      
      // Add a small delay to ensure flags are processed before any mutation observer triggers
      setTimeout(() => {
        post.dataset.harmonizerRevealed = 'true';
        post.dataset.harmonizerFiltered = 'revealed';
        console.log('JeetBlock: Flags re-confirmed after delay');
      }, 10);
      
      // Add hide button immediately after restoring content
      const hideBtn = document.createElement('button');
      hideBtn.className = 'harmonizer-hide-btn';
      hideBtn.dataset.action = 'hide';
      // Keep within flow so it doesn't jump to top-right of viewport
      hideBtn.style.cssText = 'position: static; display: inline-flex; align-items: center; gap: 6px; margin: 8px 0; background: rgba(0,0,0,0.6); color: #fff; border: none; padding: 6px 8px; border-radius: 4px; cursor: pointer;';
      hideBtn.innerHTML = `
        <span class="harmonizer-arrow">▲</span>
        <span class="harmonizer-toggle-text">Hide</span>
      `;
      hideBtn.onclick = (e) => {
        e.stopPropagation();
        this.togglePost(post);
      };
      post.prepend(hideBtn);
      
      console.log('JeetBlock: Post revealed successfully with hide button');
    } else {
      console.log('JeetBlock: No show button found or action not show');
      console.log('JeetBlock: Toggle button found:', !!toggleBtn);
      if (toggleBtn) {
        console.log('JeetBlock: Toggle button action:', toggleBtn.dataset.action);
      }
    }
  }

  revealPost(post) {
    // Legacy method - redirect to togglePost
    this.togglePost(post);
  }

  collapsePost(post) {
    post.classList.add('harmonizer-collapsed');
    post.dataset.harmonizerFiltered = 'collapsed';
    const revealBtn = post.querySelector('.harmonizer-reveal-btn');
    if (revealBtn) {
      revealBtn.textContent = 'Show Content';
      revealBtn.onclick = (e) => {
        e.stopPropagation();
        this.revealPost(post);
      };
    }
  }

  filterPost(post) {
    // Universal Instagram Search Sidebar Exemption
    if (this.isInstagramSearchSidebar(post)) {
      console.log('JeetBlock: Instagram search sidebar detected in main filter - exempting from filtering');
      return;
    }
    
    // Debug: Log YouTube-specific elements
    if (post.tagName && post.tagName.toLowerCase().includes('ytd-')) {
      console.log('JeetBlock: filterPost called on YouTube element:', post.tagName, 'harmonizerProcessed:', post.dataset.harmonizerProcessed, 'harmonizerFiltered:', post.dataset.harmonizerFiltered);
    }
    
    if (post.dataset.harmonizerFiltered) {
      console.log('JeetBlock: Post already processed, skipping');
      return; // Already processed
    }
    if (post.dataset.harmonizerProcessed) {
      console.log('JeetBlock: Post already processed by specific observer, skipping');
      return; // Already processed by platform-specific observer (e.g. YouTube)
    }
    if (post.dataset.harmonizerRevealed === 'true') {
      console.log('JeetBlock: Post manually revealed, skipping re-filter');
      return; // Manually revealed by user
    }
    
    // IMPORTANT: Set processing flag BEFORE checking to prevent double-filtering
    post.dataset.harmonizerProcessed = 'processing';
    
    if (this.shouldFilterPost(post)) {
      this.applyFilter(post);
      // Mark as fully processed
      post.dataset.harmonizerProcessed = 'true';
    } else {
      // If not filtering, clear the processing flag
      delete post.dataset.harmonizerProcessed;
    }
  }

  filterExistingPosts() {
    const selector = this.getPostSelectors();
    // Skip if selector is empty (e.g., for YouTube which has dedicated observers)
    if (!selector || selector.trim() === '') {
      return;
    }
    const posts = document.querySelectorAll(selector);
    posts.forEach(post => this.filterPost(post));
  }

  setupMutationObserver() {
    const observer = new MutationObserver((mutations) => {
      const selector = this.getPostSelectors();
      // Skip if selector is empty (e.g., for YouTube which has dedicated observers)
      if (!selector || selector.trim() === '') {
        return;
      }
      
      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            // Universal Instagram Search Sidebar Exemption
            if (this.isInstagramSearchSidebar(node)) {
              console.log('JeetBlock: Instagram search sidebar detected in mutation observer - exempting from filtering');
              return;
            }
            
            // Check if the added node is a post
            if (node.matches && node.matches(selector)) {
              console.log('JeetBlock: Mutation observer found new post:', node);
              this.filterPost(node);
            }
            
            // Check for posts within the added node
            const posts = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            if (posts.length > 0) {
              console.log('JeetBlock: Mutation observer found', posts.length, 'posts within node:', node);
            }
            posts.forEach(post => this.filterPost(post));
          }
        });
      });
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  // Public methods for popup communication
  updateSettings(settings) {
    const wasEnabled = this.isEnabled;
    
    if (settings.filterMode !== undefined) this.filterMode = settings.filterMode;
    if (settings.isEnabled !== undefined) this.isEnabled = settings.isEnabled;
    if (settings.customFilters !== undefined) this.customFilters = settings.customFilters;
    if (settings.allowList !== undefined) this.allowList = settings.allowList;
    if (settings.blockList !== undefined) this.blockList = settings.blockList;
    if (settings.filterIndian !== undefined) this.filterIndian = settings.filterIndian;
    if (settings.filterMuslim !== undefined) this.filterMuslim = settings.filterMuslim;
    if (settings.filterJewish !== undefined) this.filterJewish = settings.filterJewish;
    if (settings.filterLGBTQIA !== undefined) this.filterLGBTQIA = settings.filterLGBTQIA;
    if (settings.filterUkraine !== undefined) this.filterUkraine = settings.filterUkraine;
    if (settings.filterPalestine !== undefined) this.filterPalestine = settings.filterPalestine;
    if (settings.filterFacebook !== undefined) this.filterFacebook = settings.filterFacebook;
    if (settings.filterTwitter !== undefined) this.filterTwitter = settings.filterTwitter;
    if (settings.filterInstagram !== undefined) this.filterInstagram = settings.filterInstagram;
    if (settings.filterLinkedIn !== undefined) this.filterLinkedIn = settings.filterLinkedIn;
    if (settings.filterTikTok !== undefined) this.filterTikTok = settings.filterTikTok;
    if (settings.filterReddit !== undefined) this.filterReddit = settings.filterReddit;
    if (settings.filterYouTube !== undefined) this.filterYouTube = settings.filterYouTube;
    
    console.log('JeetBlock: Settings updated in content script:', {
      isEnabled: this.isEnabled,
      allowList: this.allowList,
      blockList: this.blockList,
      filterIndian: this.filterIndian,
      filterMuslim: this.filterMuslim,
      filterJewish: this.filterJewish,
      filterLGBTQIA: this.filterLGBTQIA,
      filterUkraine: this.filterUkraine,
      filterPalestine: this.filterPalestine
    });
    
    // If extension was just disabled, clear all filters
    if (wasEnabled && !this.isEnabled) {
      this.clearAllFilters();
    } else {
      // Re-filter all posts with new settings
      this.refilterAllPosts();
    }
  }

  refilterAllPosts() {
    const posts = document.querySelectorAll(this.getPostSelectors());
    posts.forEach(post => {
      // Remove existing filtering
      post.classList.remove('harmonizer-collapsed');
      post.style.display = '';
      delete post.dataset.harmonizerFiltered;
      
      // Remove reveal buttons
      const revealBtn = post.querySelector('.harmonizer-reveal-btn');
      if (revealBtn) revealBtn.remove();
      
      // Re-apply filtering
      this.filterPost(post);
      // For Reddit, if we previously scoped only the comment body, restore children visibility
      if (post.dataset.reddit === 'true') {
        const childrenSlots = post.querySelectorAll('[slot^="children-"]');
        childrenSlots.forEach((slot) => { slot.style.display = ''; });
      }
    });
  }

  // Clear all existing filters when extension is disabled
  clearAllFilters() {
    console.log('JeetBlock: Clearing all filters - extension disabled');
    
    // Find all filtered posts and restore them
    const filteredPosts = document.querySelectorAll('[data-harmonizer-filtered]');
    filteredPosts.forEach(post => {
      // Remove all harmonizer classes and data attributes
      post.classList.remove('harmonizer-collapsed');
      post.removeAttribute('data-harmonizer-filtered');
      post.removeAttribute('data-original-height');
      post.removeAttribute('data-original-content');
      
      // Remove harmonizer elements
      const harmonizerElements = post.querySelectorAll('.harmonizer-reveal-btn, .harmonizer-hidden-message, .harmonizer-toggle-btn');
      harmonizerElements.forEach(el => el.remove());
      
      // Restore original content if it was stored
      if (post.dataset.originalContent) {
        post.innerHTML = post.dataset.originalContent;
        post.removeAttribute('data-original-content');
      }
      
      // Reset any TikTok-specific filtering
      if (post.dataset.harmonizerFiltered === 'tiktok-blurred') {
        post.style.filter = '';
        post.style.transition = '';
        post.removeAttribute('data-harmonizer-filtered');
      }
    });
  }

  // ========================= YouTube Replies (Scoped) =========================
  setupYouTubeRepliesObserver() {
    try {
      const selector = 'ytd-comment-view-model';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterYouTubeReply(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing comments
      document.querySelectorAll(selector).forEach((el) => this.filterYouTubeReply(el));
    } catch (e) {
      console.warn('JeetBlock: YT-Reply observer error', e);
    }
  }

  filterYouTubeReply(commentEl) {
    try {
      if (!this.isEnabled || !this.filterYouTube) return;
      if (!commentEl || commentEl.dataset.harmonizerProcessed === 'true') return;

      commentEl.dataset.harmonizerProcessed = 'true';
      commentEl.dataset.ytReply = 'true'; // Mark as YouTube reply

      let username = '';
      const authorLink = commentEl.querySelector('#author-text');
      if (authorLink) {
        const href = authorLink.getAttribute('href');
        const match = href.match(/\/@([a-zA-Z0-9_.]+)/);
        if (match && match[1]) {
          username = match[1];
        } else if (authorLink.textContent) {
          username = authorLink.textContent.replace('@', '').trim();
        }
      }

      if (!username) {
        console.log('JeetBlock: YT-Comment: No username found — skipped');
        return;
      }

      const contentEl = commentEl.querySelector('#content-text, yt-attributed-string');
      const contentText = contentEl ? (contentEl.textContent || '').trim() : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(commentEl, 'youtube-reply'); // Apply filter with specific type
        console.log(`JeetBlock: YT-Comment: Filtering reply by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: YT-Comment error', e);
    }
  }

  // ========================= LinkedIn Replies (Scoped) =========================
  setupLinkedInRepliesObserver() {
    try {
      const selector = 'section.comment';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterLinkedInReply(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing comments
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInReply(el));
    } catch (e) {
      console.warn('JeetBlock: LI-Reply observer error', e);
    }
  }

  filterLinkedInReply(commentEl) {
    try {
      if (!this.isEnabled || !this.filterLinkedIn) return;
      if (!commentEl || commentEl.dataset.harmonizerProcessed === 'true') return;

      commentEl.dataset.harmonizerProcessed = 'true';
      commentEl.dataset.liReply = 'true'; // Mark as LinkedIn reply

      let username = '';
      const authorLink = commentEl.querySelector('.comment__author');
      if (authorLink && authorLink.textContent) {
        username = authorLink.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: LI-Comment: No username found — skipped');
        return;
      }

      const contentEl = commentEl.querySelector('.comment__text, .attributed-text-segment-list__content');
      const contentText = contentEl ? (contentEl.textContent || '').trim() : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(commentEl, 'linkedin-reply'); // Apply filter with specific type
        console.log(`JeetBlock: LI-Comment: Filtering reply by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: LI-Comment error', e);
    }
  }

  // ========================= LinkedIn Main Posts (Scoped) =========================
  setupLinkedInPostsObserver() {
    try {
      console.log('JeetBlock: Setting up LinkedIn posts observer');
      const selector = 'article.main-feed-activity-card';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterLinkedInPost(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing posts
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInPost(el));
    } catch (e) {
      console.warn('JeetBlock: LI-Post observer error', e);
    }
  }

  filterLinkedInPost(postEl) {
    try {
      if (!this.isEnabled || !this.filterLinkedIn) return;
      if (!postEl || postEl.dataset.harmonizerProcessed === 'true') return;

      postEl.dataset.harmonizerProcessed = 'true';
      postEl.dataset.liPost = 'true'; // Mark as LinkedIn post

      let username = '';
      // Look for the main author name in the entity lockup - try multiple selectors
      const authorLink = postEl.querySelector('[data-test-id="main-feed-activity-card__entity-lockup"] a[href*="/company/"]') ||
                        postEl.querySelector('a[href*="/company/"][data-tracking-control-name*="feed-actor-name"]') ||
                        postEl.querySelector('a[href*="/company/"]');
      if (authorLink && authorLink.textContent) {
        username = authorLink.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: LI-Post: No username found — skipped');
        return;
      }

      console.log('JeetBlock: LI-Post: Found username:', username);

      const contentEl = postEl.querySelector('[data-test-id="main-feed-activity-card__commentary"]');
      const contentText = contentEl ? (contentEl.textContent || '').trim() : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(postEl, 'linkedin-post'); // Apply filter with specific type
        console.log(`JeetBlock: LI-Post: Filtering post by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: LI-Post error', e);
    }
  }

  // ========================= LinkedIn Employee Listings (Scoped) =========================
  setupLinkedInEmployeesObserver() {
    try {
      const selector = 'li a[href*="/in/"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterLinkedInEmployee(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing employee listings
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInEmployee(el));
    } catch (e) {
      console.warn('JeetBlock: LI-Employee observer error', e);
    }
  }

  filterLinkedInEmployee(employeeEl) {
    try {
      if (!this.isEnabled || !this.filterLinkedIn) return;
      if (!employeeEl || employeeEl.dataset.harmonizerProcessed === 'true') return;

      employeeEl.dataset.harmonizerProcessed = 'true';
      employeeEl.dataset.liEmployee = 'true'; // Mark as LinkedIn employee

      let username = '';
      // Look for the employee name in the title
      const nameEl = employeeEl.querySelector('.base-main-card__title');
      if (nameEl && nameEl.textContent) {
        username = nameEl.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: LI-Employee: No name found — skipped');
        return;
      }

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      if (shouldFilter) {
        this.applyFilter(employeeEl, 'linkedin-employee'); // Apply filter with specific type
        console.log(`JeetBlock: LI-Employee: Filtering employee ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: LI-Employee error', e);
    }
  }

  // ========================= LinkedIn Reshared Posts (Scoped) =========================
  setupLinkedInResharesObserver() {
    try {
      console.log('JeetBlock: Setting up LinkedIn reshared posts observer');
      const selector = 'article.feed-reshare-content';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterLinkedInReshare(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing reshared posts
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInReshare(el));
    } catch (e) {
      console.warn('JeetBlock: LI-Reshare observer error', e);
    }
  }

  filterLinkedInReshare(reshareEl) {
    try {
      if (!this.isEnabled || !this.filterLinkedIn) return;
      if (!reshareEl || reshareEl.dataset.harmonizerProcessed === 'true') return;

      reshareEl.dataset.harmonizerProcessed = 'true';
      reshareEl.dataset.liReshare = 'true'; // Mark as LinkedIn reshare

      let username = '';
      // Look for the reshared post author name in the entity lockup
      const authorLink = reshareEl.querySelector('[data-test-id="feed-reshare-content__entity-lockup"] a[href*="/company/"]') ||
                        reshareEl.querySelector('a[href*="/company/"][data-tracking-control-name*="reshare_feed-actor-name"]') ||
                        reshareEl.querySelector('a[href*="/company/"]');
      if (authorLink && authorLink.textContent) {
        username = authorLink.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: LI-Reshare: No username found — skipped');
        return;
      }

      console.log('JeetBlock: LI-Reshare: Found username:', username);

      const contentEl = reshareEl.querySelector('[data-test-id="feed-reshare-content__commentary"]');
      const contentText = contentEl ? (contentEl.textContent || '').trim() : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + contentText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(reshareEl, 'linkedin-reshare'); // Apply filter with specific type
        console.log(`JeetBlock: LI-Reshare: Filtering reshare by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: LI-Reshare error', e);
    }
  }

  // ========================= LinkedIn Sidebar Messages =========================
  setupLinkedInSidebarMessagesObserver() {
    try {
      const selector = '.msg-conversation-listitem__link';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterLinkedInSidebarMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterLinkedInSidebarMessage(item));
      console.log('JeetBlock: LinkedIn sidebar messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn sidebar messages observer', e);
    }
  }

  filterLinkedInSidebarMessage(messageItem) {
    if (!this.isEnabled || !this.filterLinkedIn) return;
    try {
      if (!messageItem || messageItem.dataset.harmonizerProcessed === 'true') return;

      // Extract username from the participant names span
      const usernameEl = messageItem.querySelector('.msg-conversation-listitem__participant-names span');
      if (!usernameEl) return;

      const username = usernameEl.textContent.trim();
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      let reason = '';

      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reason = 'Jewish name';
      } else if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reason = 'Muslim name';
      } else if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reason = 'Indian name';
      }

      if (!shouldFilter) return;

      messageItem.dataset.harmonizerProcessed = 'true';
      messageItem.dataset.liSidebarMessage = 'true';
      this.applyFilter(messageItem, 'linkedin-sidebar-message');
      console.log(`JeetBlock: LinkedIn Sidebar Message filtered: ${username} - ${reason}`);
    } catch (e) {
      console.warn('JeetBlock: LinkedIn Sidebar Message filter error', e);
    }
  }

  // ========================= LinkedIn Chat Dialog =========================
  setupLinkedInChatDialogObserver() {
    try {
      const selector = 'li.msg-s-event-listitem';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterLinkedInChatMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterLinkedInChatMessage(item));
      console.log('JeetBlock: LinkedIn chat dialog observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn chat dialog observer', e);
    }
  }

  filterLinkedInChatMessage(messageItem) {
    if (!this.isEnabled || !this.filterLinkedIn) return;
    try {
      if (!messageItem || messageItem.dataset.harmonizerProcessed === 'true') return;

      // Skip if this is part of the message input form
      if (messageItem.closest('.msg-form')) return;

      // Extract username from the message group name
      const usernameEl = messageItem.querySelector('.msg-s-message-group__name');
      if (!usernameEl) return;

      const username = usernameEl.textContent.trim();
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      let reason = '';

      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reason = 'Jewish name';
      } else if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reason = 'Muslim name';
      } else if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reason = 'Indian name';
      }

      if (!shouldFilter) return;

      messageItem.dataset.harmonizerProcessed = 'true';
      messageItem.dataset.liChatMessage = 'true';
      this.applyFilter(messageItem, 'linkedin-chat-message');
      console.log(`JeetBlock: LinkedIn Chat Message filtered: ${username} - ${reason}`);
    } catch (e) {
      console.warn('JeetBlock: LinkedIn Chat Message filter error', e);
    }
  }

  // ========================= LinkedIn Main Messages Page =========================
  setupLinkedInMainMessagesObserver() {
    try {
      const selector = 'li.msg-conversation-listitem';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterLinkedInMainMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterLinkedInMainMessage(item));
      console.log('JeetBlock: LinkedIn main messages observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn main messages observer', e);
    }
  }

  filterLinkedInMainMessage(messageItem) {
    if (!this.isEnabled || !this.filterLinkedIn) return;
    try {
      if (!messageItem || messageItem.dataset.harmonizerProcessed === 'true') return;

      // Extract username from the participant names span
      const usernameEl = messageItem.querySelector('.msg-conversation-listitem__participant-names span');
      if (!usernameEl) return;

      const username = usernameEl.textContent.trim();
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      let reason = '';

      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reason = 'Jewish name';
      } else if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reason = 'Muslim name';
      } else if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reason = 'Indian name';
      }

      if (!shouldFilter) return;

      messageItem.dataset.harmonizerProcessed = 'true';
      messageItem.dataset.liMainMessage = 'true';
      this.applyFilter(messageItem, 'linkedin-main-message');
      console.log(`JeetBlock: LinkedIn Main Message filtered: ${username} - ${reason}`);
    } catch (e) {
      console.warn('JeetBlock: LinkedIn Main Message filter error', e);
    }
  }

  // ========================= LinkedIn Main Messaging Window =========================
  setupLinkedInMainMessagingObserver() {
    try {
      const selector = 'li.msg-s-event-listitem';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterLinkedInMainMessagingMessage(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterLinkedInMainMessagingMessage(item));
      console.log('JeetBlock: LinkedIn main messaging observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn main messaging observer', e);
    }
  }

  filterLinkedInMainMessagingMessage(messageItem) {
    if (!this.isEnabled || !this.filterLinkedIn) return;
    try {
      if (!messageItem || messageItem.dataset.harmonizerProcessed === 'true') return;

      // Skip if this is part of the message input form
      if (messageItem.closest('.msg-form')) return;

      // Extract username from the message group name
      const usernameEl = messageItem.querySelector('.msg-s-message-group__name');
      if (!usernameEl) return;

      const username = usernameEl.textContent.trim();
      if (!username) return;

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      let reason = '';

      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reason = 'Jewish name';
      } else if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reason = 'Muslim name';
      } else if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reason = 'Indian name';
      }

      if (!shouldFilter) return;

      messageItem.dataset.harmonizerProcessed = 'true';
      messageItem.dataset.liMainMessagingMessage = 'true';
      this.applyFilter(messageItem, 'linkedin-main-messaging-message');
      console.log(`JeetBlock: LinkedIn Main Messaging Message filtered: ${username} - ${reason}`);
    } catch (e) {
      console.warn('JeetBlock: LinkedIn Main Messaging Message filter error', e);
    }
  }

  // ========================= LinkedIn Conversation Detail (Header) =========================
  setupLinkedInConversationHeaderObserver() {
    try {
      console.log('JeetBlock: Setting up LinkedIn conversation header observer');
      // Header shows the participant's name in the main thread area
      const selector = '.msg-title-bar .msg-entity-lockup__entity-title, a.msg-thread__link-to-profile, .msg-s-profile-card .artdeco-entity-lockup__title';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((el) => this.filterLinkedInConversationDetail(el)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInConversationDetail(el));
      console.log('JeetBlock: LinkedIn conversation header observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn conversation header observer', e);
    }
  }

  filterLinkedInConversationDetail(headerEl) {
    if (!this.isEnabled || !this.filterLinkedIn) return;
    try {
      if (!headerEl) return;
      // Find the overall conversation wrapper to collapse
      const convoWrapper = headerEl.closest('.msg-convo-wrapper') || document.querySelector('.msg-convo-wrapper');
      if (!convoWrapper || convoWrapper.dataset.harmonizerProcessed === 'true') return;

      // Gather candidate strings for name/script/emoji checks
      const candidates = [];
      // Header title text
      const h2 = convoWrapper.querySelector('.msg-title-bar .msg-entity-lockup__entity-title');
      if (h2 && h2.textContent) candidates.push(h2.textContent.trim());
      // Profile link text
      const profileLink = convoWrapper.querySelector('a.msg-thread__link-to-profile .msg-entity-lockup__entity-title, a.msg-thread__link-to-profile span.truncate');
      if (profileLink && profileLink.textContent) candidates.push(profileLink.textContent.trim());
      // Any alt texts (profile images include full name)
      convoWrapper.querySelectorAll('img[alt]').forEach(img => {
        const alt = img.getAttribute('alt');
        if (alt) candidates.push(alt.trim());
      });
      // Any aria-labels mentioning conversation actions
      convoWrapper.querySelectorAll('[aria-label]').forEach(el => {
        const label = el.getAttribute('aria-label');
        if (label) candidates.push(label.trim());
      });

      if (!candidates.length) return;

      let shouldFilter = false;
      const reasons = [];
      const joinedLower = candidates.join(' ').toLowerCase();

      // Name-based checks (favor exact names present in UI strings)
      for (const c of candidates) {
        const lower = c.toLowerCase();
        if (this.containsJewishName(lower)) { shouldFilter = true; reasons.push('Jewish name'); break; }
        if (this.containsMuslimName(lower)) { shouldFilter = true; reasons.push('Muslim name'); break; }
        if (this.containsIndianName(lower)) { shouldFilter = true; reasons.push('Indian name'); break; }
      }

      // Script/emoji checks anywhere in the header/card UI
      if (!shouldFilter) {
        if (this.containsHebrew(joinedLower)) { shouldFilter = true; reasons.push('Hebrew script'); }
        if (!shouldFilter && (this.containsArabic(joinedLower) || this.containsPersian(joinedLower) || this.containsUrdu(joinedLower))) { shouldFilter = true; reasons.push('Arabic-family script'); }
        if (!shouldFilter && (this.containsDevanagari(joinedLower) || this.containsBengali(joinedLower) || this.containsSinhala(joinedLower))) { shouldFilter = true; reasons.push('Indian script'); }
        if (!shouldFilter && (this.containsLGBTQIAEmoji(joinedLower) || this.containsUkraineEmoji(joinedLower) || this.containsPalestineEmoji(joinedLower))) { shouldFilter = true; reasons.push('flag/emoji'); }
      }

      if (!shouldFilter) return;

      convoWrapper.dataset.harmonizerProcessed = 'true';
      convoWrapper.dataset.linkedin = 'true';
      this.applyFilter(convoWrapper, 'linkedin-conversation');
      console.log('JeetBlock: LinkedIn Conversation filtered:', { candidates: candidates.slice(0, 5), reasons });
    } catch (e) {
      console.warn('JeetBlock: LinkedIn conversation detail filter error', e);
    }
  }

  // ========================= LinkedIn Search Results (Scoped) =========================
  setupLinkedInSearchResultsObserver() {
    try {
      console.log('JeetBlock: Setting up LinkedIn search results observer');
      const selector = '[data-chameleon-result-urn]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else if (n.querySelector && n.querySelector(selector)) {
                  pending.push(...n.querySelectorAll(selector));
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => pending.forEach((el) => this.filterLinkedInSearchResult(el)));
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing search results
      document.querySelectorAll(selector).forEach((el) => this.filterLinkedInSearchResult(el));
      console.log('JeetBlock: LinkedIn search results observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize LinkedIn search results observer', e);
    }
  }

  filterLinkedInSearchResult(searchEl) {
    try {
      if (!this.isEnabled || !this.filterLinkedIn) return;
      
      searchEl.dataset.liSearch = 'true'; // Mark as LinkedIn search result
      
      // Extract username from the search result
      let username = '';
        const nameElement = searchEl.querySelector('span[dir="ltr"] span[aria-hidden="true"]');
      if (nameElement) {
        username = this.getTextWithEmojis(nameElement);
      }
      
      if (!username) {
        // Fallback: try to get name from any text content
        const textElements = searchEl.querySelectorAll('span, a, div');
        for (const el of textElements) {
          const text = this.getTextWithEmojis(el);
          if (text && text.length > 2 && text.length < 50) {
            username = text;
            break;
          }
        }
      }
      
      if (username) {
        const reasons = [];
        let shouldFilter = false;
        
        // Check Indian names
        if (this.filterIndian) {
          const matchedNames = this.getMatchedIndianNames(username);
          if (matchedNames.length > 0) {
            shouldFilter = true;
            reasons.push(`Indian names: ${matchedNames.join(', ')}`);
          }
          
          // Check for Indian scripts
          if (this.containsDevanagari(username) || this.containsBengali(username) || 
              this.containsSinhala(username) || this.containsGurmukhi(username)) {
            shouldFilter = true;
            reasons.push('Indian script');
          }
          
          // Check for Indian emojis
          if (this.containsIndianEmoji(username)) {
            shouldFilter = true;
            reasons.push('Indian emoji');
          }
        }
        
        // Check Muslim names
        if (this.filterMuslim) {
          const matchedNames = this.getMatchedMuslimNames(username);
          if (matchedNames.length > 0) {
            shouldFilter = true;
            reasons.push(`Muslim names: ${matchedNames.join(', ')}`);
          }
          
          // Check for Arabic/Persian/Urdu scripts
          if (this.containsArabic(username) || this.containsPersian(username) || this.containsUrdu(username)) {
            shouldFilter = true;
            reasons.push('Arabic/Persian/Urdu script');
          }
          
          // Check for Muslim emojis
          if (this.containsMuslimEmoji(username)) {
            shouldFilter = true;
            reasons.push('Muslim emoji');
          }
        }
        
        // Check Jewish names
        if (this.filterJewish) {
          const matchedNames = this.getMatchedJewishNames(username);
          if (matchedNames.length > 0) {
            shouldFilter = true;
            reasons.push(`Jewish names: ${matchedNames.join(', ')}`);
          }
          
          // Check for Hebrew script
          if (this.containsHebrew(username)) {
            shouldFilter = true;
            reasons.push('Hebrew script');
          }
          
          // Check for Jewish emojis
          if (this.containsJewishEmoji(username)) {
            shouldFilter = true;
            reasons.push('Jewish emoji');
          }
        }
        
        if (shouldFilter) {
          this.applyFilter(searchEl, 'linkedin-search-result');
          console.log(`JeetBlock: LinkedIn Search Result filtered: ${username} (reasons: ${reasons.join(', ')})`);
        }
      }
    } catch (e) {
      console.warn('JeetBlock: LinkedIn search result filter error', e);
    }
  }

  // ========================= YouTube Videos (Scoped) =========================
  setupYouTubeVideosObserver() {
    try {
      console.log('JeetBlock: Setting up YouTube videos observer');
      const selector = 'ytd-rich-item-renderer';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const n of m.addedNodes) {
              if (n.nodeType === 1) { // Element node
                if (n.matches && n.matches(selector)) {
                  pending.push(n);
                } else {
                  const matches = n.querySelectorAll && n.querySelectorAll(selector);
                  if (matches) pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterYouTubeVideo(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing videos
      document.querySelectorAll(selector).forEach((el) => this.filterYouTubeVideo(el));
    } catch (e) {
      console.warn('JeetBlock: YT-Video observer error', e);
    }
  }

  filterYouTubeVideo(videoEl) {
    try {
      if (!this.isEnabled || !this.filterYouTube) return;
      if (!videoEl || videoEl.dataset.harmonizerProcessed) {
        if (videoEl && videoEl.dataset.harmonizerProcessed) {
          console.log('JeetBlock: YT-Video: Already processed, skipping:', videoEl.tagName);
        }
        return;
      }

      console.log('JeetBlock: YT-Video: Processing NEW video element:', videoEl.tagName, 'ID:', videoEl.id || 'no-id');
      videoEl.dataset.harmonizerProcessed = 'true';
      videoEl.dataset.ytVideo = 'true'; // Mark as YouTube video

      let username = '';
      // Look for the channel name in the metadata
      const channelLink = videoEl.querySelector('a[href*="/@"]');
      if (channelLink) {
        username = this.getTextWithEmojis(channelLink);
      }

      if (!username) {
        console.log('JeetBlock: YT-Video: No channel name found — skipped');
        return;
      }

      console.log('JeetBlock: YT-Video: Found channel:', username);

      const titleEl = videoEl.querySelector('.yt-lockup-metadata-view-model__title');
      const titleText = titleEl ? this.getTextWithEmojis(titleEl) : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + titleText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use only username, not video title)
      if (this.containsDevanagari(usernameLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(usernameLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(usernameLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsGurmukhi(usernameLower)) {
        shouldFilter = true;
        reasons.push('Gurmukhi script');
      }
      if (this.containsIndianEmoji(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(videoEl, 'youtube-video'); // Apply filter with specific type
        console.log(`JeetBlock: YT-Video: Filtering video by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: YT-Video error', e);
    }
  }

  // ========================= YouTube Search Results (Scoped) =========================
  setupYouTubeSearchObserver() {
    try {
      console.log('JeetBlock: Setting up YouTube search observer');
      const selector = 'ytd-video-renderer';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterYouTubeSearchResult(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing search results
      document.querySelectorAll(selector).forEach((el) => this.filterYouTubeSearchResult(el));
    } catch (e) {
      console.warn('JeetBlock: YT-Search observer error', e);
    }
  }

  filterYouTubeSearchResult(videoEl) {
    try {
      if (!this.isEnabled || !this.filterYouTube) return;
      if (!videoEl || videoEl.dataset.harmonizerProcessed === 'true') return;

      videoEl.dataset.harmonizerProcessed = 'true';
      videoEl.dataset.ytSearch = 'true'; // Mark as YouTube search result

      let username = '';
      // Look for the channel name in the search result
      const channelNameEl = videoEl.querySelector('ytd-channel-name a[href*="/@"]');
      if (channelNameEl) {
        username = this.getTextWithEmojis(channelNameEl);
      }

      if (!username) {
        console.log('JeetBlock: YT-Search: No channel name found — skipped');
        return;
      }

      console.log('JeetBlock: YT-Search: Found channel:', username);

      const titleEl = videoEl.querySelector('#video-title');
      const titleText = titleEl ? this.getTextWithEmojis(titleEl) : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + titleText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text - channel name + video title)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsGurmukhi(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Gurmukhi script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(videoEl, 'youtube-search'); // Apply filter with specific type
        console.log(`JeetBlock: YT-Search: Filtering search result by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: YT-Search error', e);
    }
  }

  // ========================= TikTok Search Results (Scoped) =========================
  setupTikTokSearchObserver() {
    try {
      console.log('JeetBlock: Setting up TikTok search observer');
      const selector = 'div[data-e2e="search_top-item"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterTikTokSearchResult(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing search results
      document.querySelectorAll(selector).forEach((el) => this.filterTikTokSearchResult(el));
    } catch (e) {
      console.warn('JeetBlock: TT-Search observer error', e);
    }
  }

  filterTikTokSearchResult(searchEl) {
    try {
      if (!this.isEnabled || !this.filterTikTok) return;
      if (!searchEl) return;
      // Allow reprocessing unless already collapsed/hidden
      if (searchEl.dataset.harmonizerFiltered === 'collapsed') return;

      // Mark type, but only mark processed if we actually filter
      searchEl.dataset.ttSearch = 'true';

      // Collect candidate usernames from multiple sources
      const candidates = [];
      const uidEl = searchEl.querySelector('p[data-e2e="search-card-user-unique-id"]');
      if (uidEl && uidEl.textContent) candidates.push(uidEl.textContent.trim());
      const linkEl = searchEl.querySelector('a[data-e2e="search-card-user-link"][href*="/@"]');
      const href = linkEl ? linkEl.getAttribute('href') : '';
      if (href && href.includes('/@')) {
        const m = href.match(/\/@([^\/?#]+)/);
        if (m && m[1]) candidates.push(m[1]);
      }
      // Also derive display name from aria-label when available (e.g., "Mohammed 🇦🇪’s profile")
      const aria = linkEl ? (linkEl.getAttribute('aria-label') || '') : '';
        if (aria) {
          const m2 = aria.match(/^(.+?)'s profile/);
        if (m2 && m2[1]) candidates.push(m2[1].trim());
      }
      const captionHandles = searchEl.querySelectorAll('[data-e2e="search-card-video-caption"] a[href*="/@"]');
      captionHandles.forEach(a => {
        const t = (a.textContent || '').trim();
        if (t) candidates.push(t.replace('@', '').trim());
      });
      // Try fetching author unique id printed in the bottom info line
      const bottomUid = searchEl.querySelector('[data-e2e="search-card-user-unique-id"]');
      if (bottomUid && bottomUid.textContent) candidates.push(bottomUid.textContent.trim());

      // Include alt text of preview image for patterns like "created by Mohammed"
      const previewImg = searchEl.querySelector('img[alt]');
      if (previewImg && previewImg.alt) {
        const alt = previewImg.alt;
        // Extract @handle from alt
        const mAltHandle = alt.match(/@([A-Za-z0-9_.]+)/);
        if (mAltHandle && mAltHandle[1]) candidates.push(mAltHandle[1]);
        // Extract "created by NAME" pattern
        const mAltCreated = alt.match(/created by\s+([^\n]+?)(?:\s+with|$)/i);
        if (mAltCreated && mAltCreated[1]) candidates.push(mAltCreated[1].trim());
      }

      // Get the description content for script/emoji checks
      const descEl = searchEl.querySelector('[data-e2e="search-card-video-caption"]');
      const descText = descEl ? (descEl.textContent || '').trim() : '';
      // Also read the paired description card if it exists (search-card-desc is sibling block)
      let pairedDescText = '';
      try {
        const descBlock = searchEl.nextElementSibling;
        if (descBlock && descBlock.getAttribute('data-e2e') === 'search-card-desc') {
          pairedDescText = (descBlock.textContent || '').trim();
        }
      } catch (_) {}

      let shouldFilter = false;
      const reasons = [];

      // Name-based checks over all candidates
      for (const raw of candidates) {
        const nameLower = (raw || '').toLowerCase();
        if (!nameLower) continue;
        if (this.containsJewishName(nameLower)) { shouldFilter = true; reasons.push('Jewish name'); break; }
        if (this.containsMuslimName(nameLower)) { shouldFilter = true; reasons.push('Muslim name'); break; }
        if (this.containsIndianName(nameLower)) { shouldFilter = true; reasons.push('Indian name'); break; }
      }

      const fullLower = (candidates.join(' ') + ' ' + descText + ' ' + pairedDescText).toLowerCase();
      // Indian scripts/emojis in caption/desc
      if (!shouldFilter && this.containsDevanagari(fullLower)) { shouldFilter = true; reasons.push('Devanagari script'); }
      if (!shouldFilter && this.containsBengali(fullLower)) { shouldFilter = true; reasons.push('Bengali script'); }
      if (!shouldFilter && this.containsSinhala(fullLower)) { shouldFilter = true; reasons.push('Sinhala script'); }
      if (!shouldFilter && this.containsIndianEmoji(fullLower)) { shouldFilter = true; reasons.push('Indian emoji'); }
      // Arabic-family scripts and Muslim emoji in caption/desc
      if (!shouldFilter && this.filterMuslim) {
        const muslimChecks = [];
        if (this.containsArabic(fullLower)) muslimChecks.push('Arabic script');
        if (this.containsPersian(fullLower)) muslimChecks.push('Persian script');
        if (this.containsUrdu(fullLower)) muslimChecks.push('Urdu script');
        if (this.containsPashto && this.containsPashto(fullLower)) muslimChecks.push('Pashto script');
        if (this.containsDari && this.containsDari(fullLower)) muslimChecks.push('Dari script');
        if (this.containsMuslimEmoji(fullLower)) muslimChecks.push('Muslim emoji');
        if (muslimChecks.length) { shouldFilter = true; reasons.push(...muslimChecks); }
      }
      // LGBTQIA / Ukraine / Palestine flags in caption/desc
      if (!shouldFilter && this.filterLGBTQIA && this.containsLGBTQIAEmoji(fullLower)) { shouldFilter = true; reasons.push('LGBTQIA+ emoji'); }
      if (!shouldFilter && this.filterUkraine && this.containsUkraineEmoji(fullLower)) { shouldFilter = true; reasons.push('Ukraine flag'); }
      if (!shouldFilter && this.filterPalestine && this.containsPalestineEmoji(fullLower)) { shouldFilter = true; reasons.push('Palestine flag'); }

      if (shouldFilter) {
        this.applyFilter(searchEl, 'tiktok-search');
        // Also hide the paired description card if present
        try {
          const descBlock = searchEl.nextElementSibling;
          if (descBlock && descBlock.getAttribute('data-e2e') === 'search-card-desc') {
            descBlock.style.display = 'none';
            descBlock.dataset.harmonizerFiltered = 'collapsed';
          }
        } catch (_) {}
        searchEl.dataset.harmonizerProcessed = 'true';
        console.log('JeetBlock: TT-Search: Filtering search result', { candidates, reasons });
      } else {
        // Allow future reprocessing if our heuristics improve
        delete searchEl.dataset.harmonizerProcessed;
        delete searchEl.dataset.ttSearch;
      }
    } catch (e) {
      console.warn('JeetBlock: TT-Search error', e);
    }
  }

  // ========================= TikTok Suggested Users (Search) =========================
  setupTikTokSuggestedUsersObserver() {
    try {
      console.log('JeetBlock: Setting up TikTok suggested users observer');
      const selector = 'div[data-e2e="search-user-container"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterTikTokSuggestedUser(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing suggested users
      document.querySelectorAll(selector).forEach((el) => this.filterTikTokSuggestedUser(el));
    } catch (e) {
      console.warn('JeetBlock: TT-Suggested observer error', e);
    }
  }

  filterTikTokSuggestedUser(suggestEl) {
    try {
      if (!this.isEnabled || !this.filterTikTok) return;
      if (!suggestEl || suggestEl.dataset.harmonizerProcessed === 'true') return;

      suggestEl.dataset.harmonizerProcessed = 'true';
      suggestEl.dataset.ttSuggested = 'true';

      let username = '';
      // Preferred unique id
      const idEl = suggestEl.querySelector('p[data-e2e="search-user-unique-id"]');
      if (idEl && idEl.textContent) username = idEl.textContent.trim();
      // Fallback nickname text
      if (!username) {
        const nickEl = suggestEl.querySelector('p[data-e2e="search-user-nickname"]');
        if (nickEl && nickEl.textContent) username = nickEl.textContent.trim();
      }
      // Fallback href /@handle
      if (!username) {
        const linkEl = suggestEl.querySelector('a[data-e2e="search-user-info-container"][href*="/@"]');
        const href = linkEl ? linkEl.getAttribute('href') : '';
        if (href && href.includes('/@')) {
          const m = href.match(/\/@([^\/?#]+)/);
          if (m && m[1]) username = m[1];
        }
      }

      if (!username) {
        console.log('JeetBlock: TT-Suggested: No username found — skipped');
        return;
      }

      const usernameLower = username.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      if (this.containsJewishName(usernameLower)) { shouldFilter = true; reasons.push('Jewish name'); }
      if (this.containsMuslimName(usernameLower)) { shouldFilter = true; reasons.push('Muslim name'); }
      if (this.containsIndianName(usernameLower)) { shouldFilter = true; reasons.push('Indian name'); }

      if (shouldFilter) {
        this.applyFilter(suggestEl, 'tiktok-suggested');
        console.log(`JeetBlock: TT-Suggested: Filtering suggested user ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: TT-Suggested error', e);
    }
  }

  // ========================= TikTok Fullscreen (Profile/Video Page) =========================
  setupTikTokFullscreenObserver() {
    try {
      console.log('JeetBlock: Setting up TikTok fullscreen observer');
      // Containers that hold username and actions on fullscreen/right-rail page
      const selector = 'div[data-e2e="browse-user-avatar"], div[class*="--DivProfileWrapper"], [data-e2e="browse-username"], [data-e2e="browser-nickname"], [data-e2e="browse-video"], [data-e2e="video-desc"], div[class*="--DivVideoMeta"], div[class*="--DivVideoMetaContainer"]';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node.closest('div[class*="--DivProfileWrapper"]') || node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  matches.forEach(match => pending.push(match.closest('div[class*="--DivProfileWrapper"]') || match));
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterTikTokFullscreen(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing
      document.querySelectorAll(selector).forEach((el) => this.filterTikTokFullscreen(el.closest('div[class*="--DivProfileWrapper"]') || el));
    } catch (e) {
      console.warn('JeetBlock: TT-Fullscreen observer error', e);
    }
  }

  filterTikTokFullscreen(profileWrapper) {
    try {
      if (!this.isEnabled || !this.filterTikTok) return;
      if (!profileWrapper || profileWrapper.dataset.harmonizerProcessed) return;

      // Extract username: prefer unique id span and href /@handle
      let username = '';
      const idSpan = profileWrapper.querySelector('[data-e2e="browse-username"] *');
      if (idSpan && idSpan.textContent) username = idSpan.textContent.trim();
      const nicknameSpan = profileWrapper.querySelector('[data-e2e="browser-nickname"] *');
      const nicknameText = nicknameSpan && nicknameSpan.textContent ? nicknameSpan.textContent.trim() : '';
      // Also capture description/meta text for emoji/script checks (not for name matching)
      const descEl = document.querySelector('[data-e2e="video-desc"]');
      const descText = descEl && descEl.textContent ? descEl.textContent.trim() : '';
      const metaEl = document.querySelector('div[class*="--DivVideoMetaContainer"], div[class*="--DivVideoMeta"]');
      const metaText = metaEl && metaEl.textContent ? metaEl.textContent.trim() : '';
      if (!username) {
        // Delay briefly to let right-rail mount populate
        const attempt = parseInt(profileWrapper.dataset.ttFullTry || '0');
        if (attempt < 5) {
          profileWrapper.dataset.ttFullTry = String(attempt + 1);
          setTimeout(() => this.filterTikTokFullscreen(profileWrapper), 150);
          return;
        }
      }
      if (!username) {
        const link = profileWrapper.querySelector('a[href^="/@"]');
        const href = link ? link.getAttribute('href') : '';
        const m = href ? href.match(/\/@([^\/?#]+)/) : null;
        if (m && m[1]) username = m[1];
      }

      // Mark as processing to avoid double work but allow reprocessing if not filtered
      profileWrapper.dataset.harmonizerProcessed = 'processing';
      profileWrapper.dataset.ttFullscreen = 'true';

      const usernameLower = (username || '').toLowerCase();
      const nicknameLower = (nicknameText || '').toLowerCase();
      const fullLower = (usernameLower + ' ' + nicknameLower);
      const contentLower = (descText + ' ' + metaText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Names in handle or display name
      if (this.containsJewishName(usernameLower) || this.containsJewishName(fullLower)) { shouldFilter = true; reasons.push('Jewish name'); }
      if (this.containsMuslimName(usernameLower) || this.containsMuslimName(fullLower)) { shouldFilter = true; reasons.push('Muslim name'); }
      if (this.containsIndianName(usernameLower) || this.containsIndianName(fullLower)) { shouldFilter = true; reasons.push('Indian name'); }

      // Emoji/flag checks against username/nickname and description/meta
      if (!shouldFilter && this.filterLGBTQIA && this.containsLGBTQIAEmoji(fullLower + ' ' + contentLower)) { shouldFilter = true; reasons.push('LGBTQIA+ emoji'); }
      if (!shouldFilter && this.filterUkraine && this.containsUkraineEmoji(fullLower + ' ' + contentLower)) { shouldFilter = true; reasons.push('Ukraine flag'); }
      if (!shouldFilter && this.filterPalestine && this.containsPalestineEmoji(fullLower + ' ' + contentLower)) { shouldFilter = true; reasons.push('Palestine flag'); }
      if (!shouldFilter && this.filterMuslim) {
        if (this.containsArabic(fullLower) || this.containsPersian(fullLower) || this.containsUrdu(fullLower) ||
            this.containsArabic(contentLower) || this.containsPersian(contentLower) || this.containsUrdu(contentLower)) {
          shouldFilter = true; reasons.push('Arabic-family script');
        }
      }

      console.log('JeetBlock: TT-Fullscreen: Extracted data:', { username, nicknameText, descText, metaText, shouldFilter, reasons });

      if (!shouldFilter) { delete profileWrapper.dataset.harmonizerProcessed; return; }

      // Mark as fully processed now that we are filtering
      profileWrapper.dataset.harmonizerProcessed = 'true';

      // Hide the profile wrapper and also pause/hide the playing video container nearby
      this.applyFilter(profileWrapper, 'tiktok-fullscreen');

      // Find nearest video container and stop it
      const videoContainers = document.querySelectorAll('video');
      videoContainers.forEach(v => {
        try { v.pause(); v.currentTime = 0; v.muted = true; v.style.pointerEvents = 'none'; } catch (_) {}
      });

      // Specifically hide the fullscreen video area but keep navigation arrows visible
      try {
        const videoWrapper = document.querySelector('div[class*="--DivVideoWrapper"]');
        if (videoWrapper) {
          videoWrapper.dataset.harmonizerFiltered = 'collapsed';
          videoWrapper.style.display = 'none';
        }
        const controlContainers = document.querySelectorAll('div[class*="--DivVideoControlContainer"], div[class*="--DivVideoControlTop"], div[class*="--DivVideoControlBottom"]');
        controlContainers.forEach(el => {
          el.dataset.harmonizerFiltered = 'collapsed';
          el.style.display = 'none';
        });
        // Hide the basic player wrapper as extra safeguard
        const basicPlayerWrapper = document.querySelector('div[data-e2e="browse-video"]');
        if (basicPlayerWrapper) {
          basicPlayerWrapper.style.display = 'none';
          basicPlayerWrapper.dataset.harmonizerFiltered = 'collapsed';
        }
      } catch (_) {}

      // Optionally click next (down/right) button if present
      const nextBtn = document.querySelector('[data-e2e="arrow-right"], [data-e2e="browse-next"]');
      if (nextBtn) {
        setTimeout(() => { try { nextBtn.click(); } catch (_) {} }, 150);
      }

      console.log(`JeetBlock: TT-Fullscreen: Filtering video by ${username} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: TT-Fullscreen error', e);
    }
  }

  // ========================= TikTok Messages (Sidebar) =========================
  setupTikTokMessagesObserver() {
    try {
      console.log('JeetBlock: Setting up TikTok messages observer');
      const selector = 'div[data-e2e="chat-list-item"]';
      
      // Debug: Check what elements exist on the page
      console.log('JeetBlock: TT-Messages: Looking for elements with selector:', selector);
      const existingElements = document.querySelectorAll(selector);
      console.log('JeetBlock: TT-Messages: Found', existingElements.length, 'existing elements');
      
      // Debug: Look for any elements that might be message items
      const allDivs = document.querySelectorAll('div');
      console.log('JeetBlock: TT-Messages: Total divs on page:', allDivs.length);
      
      // Debug: Look for elements with data-e2e attributes
      const allDataE2e = document.querySelectorAll('[data-e2e]');
      console.log('JeetBlock: TT-Messages: Elements with data-e2e:', allDataE2e.length);
      Array.from(allDataE2e).slice(0, 10).forEach((el, i) => {
        console.log(`JeetBlock: TT-Messages: data-e2e[${i}]:`, el.getAttribute('data-e2e'), el.tagName, el.className);
      });
      
      // Debug: Look for elements that might contain usernames
      const elementsWithText = Array.from(document.querySelectorAll('*')).filter(el => 
        el.textContent && el.textContent.trim().length > 0 && el.textContent.trim().length < 50
      );
      console.log('JeetBlock: TT-Messages: Elements with short text content:', elementsWithText.length);
      elementsWithText.slice(0, 10).forEach((el, i) => {
        console.log(`JeetBlock: TT-Messages: text[${i}]:`, el.textContent.trim(), el.tagName, el.className);
      });
      
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterTikTokMessage(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing messages
      document.querySelectorAll(selector).forEach((el) => this.filterTikTokMessage(el));
    } catch (e) {
      console.warn('JeetBlock: TT-Messages observer error', e);
    }
  }

  filterTikTokMessage(messageEl) {
    try {
      console.log('JeetBlock: TT-Messages: Function called. isEnabled:', this.isEnabled, 'filterTikTok:', this.filterTikTok);
      
      if (!this.isEnabled || !this.filterTikTok) {
        console.log('JeetBlock: TT-Messages: Extension disabled or TikTok filter off - exiting');
        return;
      }
      
      if (!messageEl || messageEl.dataset.harmonizerProcessed === 'true') {
        console.log('JeetBlock: TT-Messages: Already processed or null element - skipping');
        return;
      }

      console.log('JeetBlock: TT-Messages: Message element outerHTML:', messageEl.outerHTML.substring(0, 500));

      let username = '';
      // Try multiple selectors for the nickname
      let nicknameEl = messageEl.querySelector('p[class*="PInfoNickname"]');
      console.log('JeetBlock: TT-Messages: Selector 1 (p[class*="PInfoNickname"]):', nicknameEl);
      if (nicknameEl) {
        console.log('JeetBlock: TT-Messages: nicknameEl.textContent:', `"${nicknameEl.textContent}"`);
        console.log('JeetBlock: TT-Messages: nicknameEl.innerText:', `"${nicknameEl.innerText}"`);
        console.log('JeetBlock: TT-Messages: nicknameEl.innerHTML:', nicknameEl.innerHTML);
      }
      
      if (!nicknameEl) {
        nicknameEl = messageEl.querySelector('.css-ncyd6r-5e6d46e3--PInfoNickname');
        console.log('JeetBlock: TT-Messages: Selector 2 (.css-ncyd6r-5e6d46e3--PInfoNickname):', nicknameEl);
        if (nicknameEl) {
          console.log('JeetBlock: TT-Messages: nicknameEl.textContent:', `"${nicknameEl.textContent}"`);
        }
      }
      
      if (!nicknameEl) {
        nicknameEl = messageEl.querySelector('p.ejfai8i9');
        console.log('JeetBlock: TT-Messages: Selector 3 (p.ejfai8i9):', nicknameEl);
        if (nicknameEl) {
          console.log('JeetBlock: TT-Messages: nicknameEl.textContent:', `"${nicknameEl.textContent}"`);
        }
      }
      
      if (nicknameEl) {
        // Try textContent first, then innerText, then innerHTML as fallback
        if (nicknameEl.textContent && nicknameEl.textContent.trim()) {
        username = nicknameEl.textContent.trim();
          console.log('JeetBlock: TT-Messages: Found username via textContent:', username);
        } else if (nicknameEl.innerText && nicknameEl.innerText.trim()) {
          username = nicknameEl.innerText.trim();
          console.log('JeetBlock: TT-Messages: Found username via innerText:', username);
        } else if (nicknameEl.innerHTML) {
          // Parse innerHTML to extract text (strip tags)
          const tempDiv = document.createElement('div');
          tempDiv.innerHTML = nicknameEl.innerHTML;
          username = tempDiv.textContent.trim();
          console.log('JeetBlock: TT-Messages: Found username via innerHTML parsing:', username);
        }
      }

      // If still no username, log all <p> elements to see what's available
      if (!username) {
        console.log('JeetBlock: TT-Messages: All <p> elements in message:');
        const allPs = messageEl.querySelectorAll('p');
        allPs.forEach((p, idx) => {
          console.log(`  P ${idx}: class="${p.className}", text="${p.textContent.substring(0, 50)}"`);
        });
      }

      if (!username) {
        console.log('JeetBlock: TT-Messages: No username found — setting up retry');
        
        // Set up a retry mechanism - TikTok uses lazy loading
        const retryCount = parseInt(messageEl.dataset.ttRetryCount || '0');
        if (retryCount < 5) {
          messageEl.dataset.ttRetryCount = (retryCount + 1).toString();
          console.log(`JeetBlock: TT-Messages: Retry ${retryCount + 1}/5 in 200ms`);
          setTimeout(() => {
            // Temporarily clear the retry flag so filterTikTokMessage can run again
            delete messageEl.dataset.ttRetryCount;
            this.filterTikTokMessage(messageEl);
          }, 200);
        } else {
          console.log('JeetBlock: TT-Messages: Max retries reached, giving up');
          messageEl.dataset.harmonizerProcessed = 'true'; // Mark as processed so we don't retry forever
        }
        return;
      }

      // Only mark as processed AFTER we successfully extract username
      messageEl.dataset.harmonizerProcessed = 'true';
      messageEl.dataset.ttMessage = 'true';

      console.log('JeetBlock: TT-Messages: Found username (raw):', username);

      // Sanitize username to remove emoji/flags and non-letters
      const cleanUsername = username.replace(/[^\p{L}\p{M}\s'-]/gu, '').trim();
      if (!cleanUsername) {
        console.log('JeetBlock: TT-Messages: Username is only emojis/symbols, skipping');
        return;
      }

      console.log('JeetBlock: TT-Messages: Clean username:', cleanUsername);

      const usernameLower = cleanUsername.toLowerCase();
      console.log('JeetBlock: TT-Messages: Lowercase username for checking:', usernameLower);
      
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories with detailed logging
      console.log('JeetBlock: TT-Messages: Filter toggles:', {
        filterMuslim: this.filterMuslim,
        filterJewish: this.filterJewish,
        filterIndian: this.filterIndian,
        filterLGBTQIA: this.filterLGBTQIA,
        filterUkraine: this.filterUkraine,
        filterPalestine: this.filterPalestine,
        customFilters: this.customFilters.length
      });

      if (this.filterMuslim) {
        const hasMuslim = this.containsMuslimName(usernameLower);
        console.log('JeetBlock: TT-Messages: Muslim name check:', hasMuslim);
        if (hasMuslim) {
          shouldFilter = true;
          reasons.push('Muslim name');
        }
      }
      
      if (this.filterJewish) {
        const hasJewish = this.containsJewishName(usernameLower);
        console.log('JeetBlock: TT-Messages: Jewish name check:', hasJewish);
        if (hasJewish) {
          shouldFilter = true;
          reasons.push('Jewish name');
        }
      }
      
      if (this.filterIndian) {
        const hasIndian = this.containsIndianName(usernameLower);
        console.log('JeetBlock: TT-Messages: Indian name check:', hasIndian);
        if (hasIndian) {
          shouldFilter = true;
          reasons.push('Indian name');
        }
      }
      
      if (this.filterLGBTQIA && this.containsLGBTQIAEmoji(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ emoji');
      }
      if (this.filterUkraine && this.containsUkraineEmoji(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine flag');
      }
      if (this.filterPalestine && this.containsPalestineEmoji(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine flag');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      console.log('JeetBlock: TT-Messages: Final decision - shouldFilter:', shouldFilter, 'reasons:', reasons);

      if (shouldFilter) {
        this.applyFilter(messageEl, 'tiktok-message');
        console.log(`JeetBlock: TT-Messages: ✅ FILTERED message from ${cleanUsername} (reasons: ${reasons.join(', ')})`);
      } else {
        console.log(`JeetBlock: TT-Messages: ❌ NOT FILTERED - ${cleanUsername} did not match any filters`);
      }
    } catch (e) {
      console.warn('JeetBlock: TT-Messages error', e);
    }
  }

  // ========================= TikTok Chat Content =========================
  setupTikTokChatContentObserver() {
    try {
      const selector = 'div[data-e2e="chat-item"]';
      const observer = new MutationObserver((mutations) => {
        const items = new Set();
        for (const m of mutations) {
          m.addedNodes.forEach((node) => {
            if (node.nodeType !== Node.ELEMENT_NODE) return;
            if (node.matches && node.matches(selector)) items.add(node);
            const inner = node.querySelectorAll ? node.querySelectorAll(selector) : [];
            inner.forEach((el) => items.add(el));
          });
        }
        if (!items.size) return;
        requestAnimationFrame(() => items.forEach((item) => this.filterTikTokChatContent(item)));
      });
      observer.observe(document.body, { childList: true, subtree: true });
      document.querySelectorAll(selector).forEach((item) => this.filterTikTokChatContent(item));
      console.log('JeetBlock: TikTok chat content observer initialized');
    } catch (e) {
      console.warn('JeetBlock: Failed to initialize TikTok chat content observer', e);
    }
  }

  filterTikTokChatContent(chatItem) {
    if (!this.isEnabled || !this.filterTikTok) return;
    try {
      if (!chatItem || chatItem.dataset.harmonizerProcessed === 'true') return;

      // Skip if this is the input area
      if (chatItem.closest('[data-e2e="message-input-area"]')) return;

      // Extract username from the chat header or nickname
      let username = '';
      const nicknameEl = chatItem.querySelector('[data-e2e="chat-nickname"]');
      if (nicknameEl) {
        username = nicknameEl.textContent.trim();
      } else {
        // Fallback: look for username in the header area
        const headerEl = chatItem.closest('.css-1f52bpy-5e6d46e3--DivChatHeader');
        if (headerEl) {
          const headerNickname = headerEl.querySelector('[data-e2e="chat-nickname"]');
          if (headerNickname) username = headerNickname.textContent.trim();
        }
      }

      if (!username) return;

      // Extract message content
      const contentEl = chatItem.querySelector('p[class*="PText"]');
      const content = contentEl ? contentEl.textContent.trim() : '';

      // Sanitize username to remove emoji/flags and non-letters
      const cleanUsername = username.replace(/[^\p{L}\p{M}\s'-]/gu, '').trim();
      if (!cleanUsername) return;

      const usernameLower = cleanUsername.toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Check all filter categories
      if (this.filterMuslim && this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.filterJewish && this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.filterIndian && this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }
      if (this.filterLGBTQIA && this.containsLGBTQIAContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('LGBTQIA+ content');
      }
      if (this.filterUkraine && this.containsUkraineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Ukraine content');
      }
      if (this.filterPalestine && this.containsPalestineContent(usernameLower)) {
        shouldFilter = true;
        reasons.push('Palestine content');
      }
      if (this.customFilters.length > 0 && this.containsCustomFilter(usernameLower)) {
        shouldFilter = true;
        reasons.push('Custom filter');
      }

      if (!shouldFilter) return;

      // Prefer filtering the chat main content area to avoid covering input
      let targetEl = chatItem;
      const mainContainer = document.getElementById('main-content-messages');
      if (mainContainer) {
        const chatMain = mainContainer.querySelector('div[class*="DivChatMain"]');
        if (chatMain) targetEl = chatMain;
      }

      targetEl.dataset.harmonizerProcessed = 'true';
      targetEl.dataset.tikTokChatContent = 'true';
      this.applyFilter(targetEl, 'tiktok-chat-content');
      console.log(`JeetBlock: TikTok Chat Content filtered: ${cleanUsername} (reasons: ${reasons.join(', ')})`);
    } catch (e) {
      console.warn('JeetBlock: TikTok Chat Content filter error', e);
    }
  }

  // ========================= YouTube Sidebar (Scoped) =========================
  setupYouTubeSidebarObserver() {
    try {
      console.log('JeetBlock: Setting up YouTube sidebar observer');
      const selector = 'yt-lockup-view-model';
      const observer = new MutationObserver((mutations) => {
        const pending = [];
        for (const m of mutations) {
          if (m.type === 'childList') {
            for (const node of m.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                if (node.matches && node.matches(selector)) {
                  pending.push(node);
                } else if (node.querySelectorAll) {
                  const matches = node.querySelectorAll(selector);
                  pending.push(...matches);
                }
              }
            }
          }
        }
        if (pending.length > 0) {
          requestAnimationFrame(() => {
            pending.forEach((el) => this.filterYouTubeSidebarItem(el));
          });
        }
      });
      observer.observe(document.body, { childList: true, subtree: true });
      // Process existing sidebar items
      document.querySelectorAll(selector).forEach((el) => this.filterYouTubeSidebarItem(el));
    } catch (e) {
      console.warn('JeetBlock: YT-Sidebar observer error', e);
    }
  }

  filterYouTubeSidebarItem(lockupEl) {
    try {
      if (!this.isEnabled || !this.filterYouTube) return;
      if (!lockupEl || lockupEl.dataset.harmonizerProcessed === 'true') return;

      // IMPORTANT: Skip if this is inside a homepage video container (ytd-rich-item-renderer)
      // Those are handled by the YouTube Videos observer to avoid double-filtering
      const isInsideHomepageVideo = lockupEl.closest('ytd-rich-item-renderer');
      if (isInsideHomepageVideo) {
        console.log('JeetBlock: YT-Sidebar: Skipping - inside homepage video container');
        return;
      }

      lockupEl.dataset.harmonizerProcessed = 'true';
      lockupEl.dataset.ytSidebar = 'true'; // Mark as YouTube sidebar item

      let username = '';
      // Look for the channel name in the metadata
      const metadataEl = lockupEl.querySelector('.yt-content-metadata-view-model__metadata-text');
      if (metadataEl && metadataEl.textContent) {
        username = metadataEl.textContent.trim();
      }

      if (!username) {
        console.log('JeetBlock: YT-Sidebar: No channel name found — skipped');
        return;
      }

      console.log('JeetBlock: YT-Sidebar: Found channel:', username);

      const titleEl = lockupEl.querySelector('.yt-lockup-metadata-view-model__title');
      const titleText = titleEl ? (titleEl.textContent || '').trim() : '';

      const usernameLower = username.toLowerCase();
      const fullTextLower = (username + ' ' + titleText).toLowerCase();
      let shouldFilter = false;
      const reasons = [];

      // Name-based checks (use only username)
      if (this.containsJewishName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Jewish name');
      }
      if (this.containsMuslimName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Muslim name');
      }
      if (this.containsIndianName(usernameLower)) {
        shouldFilter = true;
        reasons.push('Indian name');
      }

      // Script and emoji checks (use full text)
      if (this.containsDevanagari(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Devanagari script');
      }
      if (this.containsBengali(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Bengali script');
      }
      if (this.containsSinhala(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Sinhala script');
      }
      if (this.containsIndianEmoji(fullTextLower)) {
        shouldFilter = true;
        reasons.push('Indian emoji');
      }

      if (shouldFilter) {
        this.applyFilter(lockupEl, 'youtube-sidebar'); // Apply filter with specific type
        console.log(`JeetBlock: YT-Sidebar: Filtering sidebar item by ${username} (reasons: ${reasons.join(', ')})`);
      }
    } catch (e) {
      console.warn('JeetBlock: YT-Sidebar error', e);
    }
  }

  // Special filter for YouTube sidebar items that preserves thumbnails
  applyYouTubeSidebarFilter(lockupEl) {
    try {
      console.log('JeetBlock: Applying YouTube sidebar filter to:', lockupEl);
      
      // Store original content
      if (!lockupEl.dataset.originalContent) {
        lockupEl.dataset.originalContent = lockupEl.innerHTML;
      }
      
      // Completely hide the content (same as regular filter)
      lockupEl.classList.add('harmonizer-collapsed');
      lockupEl.dataset.harmonizerFiltered = 'collapsed';
      lockupEl.dataset.youtube = 'true';
      
      // Create hidden message with Show button
      const hiddenMsg = document.createElement('div');
      hiddenMsg.className = 'harmonizer-hidden-message';
      hiddenMsg.innerHTML = `
        <div class="harmonizer-hidden-text">
          <strong>Content Filtered</strong><br>
          <span style="font-size: 12px; color: #666;">YouTube video hidden by JeetBlock</span>
        </div>
        <button class="harmonizer-toggle-btn harmonizer-reveal-btn" data-action="show" style="position: static; display: inline-flex; align-items: center; gap: 6px;">
          <span class="harmonizer-arrow">▼</span>
          <span class="harmonizer-toggle-text">Show</span>
        </button>
      `;
      
      // Replace content with hidden message
      lockupEl.innerHTML = '';
      lockupEl.appendChild(hiddenMsg);
      
      // Add click handler for show button
      const showBtn = hiddenMsg.querySelector('.harmonizer-toggle-btn');
      showBtn.onclick = (e) => {
        e.stopPropagation();
        this.togglePost(lockupEl);
      };
      
      console.log('JeetBlock: YouTube sidebar filter applied successfully');
    } catch (e) {
      console.warn('JeetBlock: YouTube sidebar filter error', e);
    }
  }
}

// Initialize the harmonizer only if licensed
let harmonizer = null;

// Check licensing before initializing
async function initializeHarmonizer() {
  try {
    // Check if we have a licensing instance available
    if (typeof JeetBlockLicensing !== 'undefined') {
      const licensing = new JeetBlockLicensing();
      const storedStatus = await licensing.getStoredLicenseStatus();
      
      console.log('JeetBlock: License status check:', storedStatus);
      
      // Only initialize if licensed
      if (storedStatus.licensed) {
        harmonizer = new SocialMediaHarmonizer();
        console.log('JeetBlock: Content script initialized - licensed');
      } else {
        console.log('JeetBlock: Content script blocked - not licensed');
        // Show a message that extension is not licensed
        showUnlicensedMessage();
      }
    } else {
      // If licensing not available, don't initialize - show error
      console.log('JeetBlock: Content script blocked - licensing not available');
      showUnlicensedMessage();
    }
  } catch (error) {
    console.error('JeetBlock: Error checking license in content script:', error);
    // On error, don't initialize
    showUnlicensedMessage();
  }
}

// Show unlicensed message
function showUnlicensedMessage() {
  // Remove any existing notifications first
  const existingNotifications = document.querySelectorAll('[data-jeetblock-notification]');
  existingNotifications.forEach(notification => notification.remove());
  
  // Create a subtle notification that the extension is not licensed
  const notification = document.createElement('div');
  notification.setAttribute('data-jeetblock-notification', 'true');
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: #ff4444;
    color: white;
    padding: 12px 16px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    z-index: 10000;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;
  notification.textContent = 'JeetBlock: Extension not licensed. Please activate in popup.';
  document.body.appendChild(notification);
  
  // Remove after 5 seconds
  setTimeout(() => {
    if (notification.parentNode) {
      notification.parentNode.removeChild(notification);
    }
  }, 5000);
}

// Show success message
function showSuccessMessage() {
  // Remove any existing notifications first
  const existingNotifications = document.querySelectorAll('[data-jeetblock-notification]');
  existingNotifications.forEach(notification => notification.remove());
  
  // Create a success notification
  const notification = document.createElement('div');
  notification.setAttribute('data-jeetblock-notification', 'true');
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: #28a745;
    color: white;
    padding: 12px 16px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    z-index: 10000;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;
  notification.textContent = 'JeetBlock: Extension activated successfully!';
  document.body.appendChild(notification);
  
  // Remove after 3 seconds
  setTimeout(() => {
    if (notification.parentNode) {
      notification.parentNode.removeChild(notification);
    }
  }, 3000);
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initializeHarmonizer);
} else {
  initializeHarmonizer();
}

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'updateSettings') {
    if (harmonizer) {
      harmonizer.updateSettings(request.settings);
      sendResponse({success: true});
    } else {
      sendResponse({success: false, error: 'Extension not licensed'});
    }
  } else if (request.action === 'reloadSettings') {
    if (harmonizer) {
      harmonizer.loadSettings().then(() => {
        sendResponse({success: true});
      });
      return true; // Keep message channel open for async response
    } else {
      sendResponse({success: false, error: 'Extension not licensed'});
    }
  } else if (request.action === 'getSettings') {
    if (harmonizer) {
      sendResponse({
        filterMode: harmonizer.filterMode,
        isEnabled: harmonizer.isEnabled,
        customFilters: harmonizer.customFilters,
        filterIndian: harmonizer.filterIndian,
        filterMuslim: harmonizer.filterMuslim
      });
    } else {
      sendResponse({success: false, error: 'Extension not licensed'});
    }
  } else if (request.action === 'reinitialize') {
    // Allow popup to force reinitialization after license check
    console.log('JeetBlock: Received reinitialize message from popup');
    initializeHarmonizer().then(() => {
      console.log('JeetBlock: Reinitialization complete, harmonizer:', harmonizer !== null);
      sendResponse({success: true, licensed: harmonizer !== null});
    });
    return true; // Keep message channel open for async response
  }
}); 