// JeetBlock Licensing Service
class JeetBlockLicensing {
  constructor() {
    this.deviceId = null;
    this.isLicensed = false;
    this.licenseKey = null;
    this.licenseCheckUrl = 'https://payment.jeetblock.org';
    this.pollingInterval = null;
    this.pollingTimeout = null;
    this.maxPollingTime = 5 * 60 * 1000; // 5 minutes
    this.pollingStartTime = null;
    this.devMode = false; // Dev mode bypass flag
    this.productTier = 'JeetBlock Basic'; // Product tier for unique device IDs
    this.storageKey = 'jeetblock_device_id_basic'; // Unique storage key per version
    
    // Load saved endpoint if available
    this.loadSavedEndpoint();
  }

  async loadSavedEndpoint() {
    try {
      const result = await chrome.storage.local.get(['jeetblock_auth_endpoint']);
      if (result.jeetblock_auth_endpoint) {
        this.licenseCheckUrl = result.jeetblock_auth_endpoint;
      }
    } catch (error) {
      console.warn('Could not load saved endpoint:', error);
    }
  }

  // Generate a unique device fingerprint
  async generateDeviceId() {
    try {
      // Create a fingerprint from browser and device characteristics
      const fingerprint = await this.createFingerprint();
      this.deviceId = `JEETBLOCK-${fingerprint}`;
      
      // Store in localStorage for persistence with version-specific key
      localStorage.setItem(this.storageKey, this.deviceId);
      
      return this.deviceId;
    } catch (error) {
      console.error('Error generating device ID:', error);
      // Fallback to a simple ID if fingerprinting fails
      this.deviceId = `JEETBLOCK-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      localStorage.setItem(this.storageKey, this.deviceId);
      return this.deviceId;
    }
  }

  // Create a device fingerprint
  async createFingerprint() {
    // Use full user agent as primary differentiator - each browser has unique UA
    const ua = navigator.userAgent;
    
    // Detect browser-specific APIs to differentiate browsers with similar UAs
    let browserIdentifier = 'unknown';
    try {
      // Check for Brave-specific API
      if (navigator.brave && typeof navigator.brave.isBrave === 'function') {
        const isBrave = await navigator.brave.isBrave();
        if (isBrave) {
          browserIdentifier = 'brave';
        }
      }
      // Check for Comet browser (may have specific properties)
      if (browserIdentifier === 'unknown' && (navigator.comet || window.comet)) {
        browserIdentifier = 'comet';
      }
      // Check for other browser-specific APIs
      if (browserIdentifier === 'unknown' && typeof browser !== 'undefined' && browser.runtime) {
        browserIdentifier = 'firefox';
      }
    } catch (e) {
      // If async check fails, try sync checks
      if (navigator.brave) {
        browserIdentifier = 'brave';
      } else if (navigator.comet || window.comet) {
        browserIdentifier = 'comet';
      } else if (typeof browser !== 'undefined' && browser.runtime) {
        browserIdentifier = 'firefox';
      }
    }
    
    const components = [
      ua, // Full user agent - PRIMARY differentiator for each browser
      browserIdentifier, // Browser-specific API identifier (differentiates Brave/Comet)
      this.productTier, // Product tier (Basic vs Ultra) for uniqueness
      navigator.language,
      screen.width + 'x' + screen.height,
      screen.colorDepth,
      new Date().getTimezoneOffset(),
      navigator.hardwareConcurrency,
      navigator.deviceMemory || 'unknown',
      navigator.platform,
      navigator.cookieEnabled,
      navigator.doNotTrack,
      navigator.maxTouchPoints || 'unknown'
    ];

    // Create a hash from the components
    const fingerprintString = components.join('|');
    const encoder = new TextEncoder();
    const data = encoder.encode(fingerprintString);
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    
    return hashHex.substring(0, 16).toUpperCase(); // Return first 16 characters
  }

  // Get stored device ID or generate new one
  async getDeviceId() {
    if (this.deviceId) {
      return this.deviceId;
    }

    // Check localStorage first with version-specific key
    let storedId = localStorage.getItem(this.storageKey);
    
    // Also check chrome.storage.local as backup with version-specific key
    if (!storedId) {
      try {
        const result = await chrome.storage.local.get([this.storageKey]);
        if (result[this.storageKey]) {
          storedId = result[this.storageKey];
          // Sync to localStorage for consistency
          localStorage.setItem(this.storageKey, storedId);
        }
      } catch (error) {
        console.warn('JeetBlock: Could not check chrome.storage.local for device ID:', error);
      }
    }
    
    if (storedId) {
      this.deviceId = storedId;
      // Sync to chrome.storage.local if needed
      try {
        await chrome.storage.local.set({ [this.storageKey]: storedId });
      } catch (error) {
        console.warn('JeetBlock: Could not sync device ID to chrome.storage.local:', error);
      }
      return this.deviceId;
    }

    // Generate new ID if none found
    const newId = await this.generateDeviceId();
    // Sync to chrome.storage.local
    try {
      await chrome.storage.local.set({ [this.storageKey]: newId });
    } catch (error) {
      console.warn('JeetBlock: Could not sync new device ID to chrome.storage.local:', error);
    }
    return newId;
  }

  // Check license status
  async checkLicense() {
    try {
      // Check dev mode first
      if (this.devMode) {
        console.log('JeetBlock: Dev mode enabled - bypassing license check');
        this.isLicensed = true;
        this.licenseKey = 'DEV-MODE-BYPASS';
        
        // Store dev mode status
        localStorage.setItem('jeetblock_licensed', 'true');
        localStorage.setItem('jeetblock_license_key', 'DEV-MODE-BYPASS');
        localStorage.setItem('jeetblock_dev_mode', 'true');
        localStorage.setItem('jeetblock_last_validation', Date.now().toString());
        
        return {
          licensed: true,
          licenseKey: 'DEV-MODE-BYPASS',
          devMode: true
        };
      }

      // Security check - if compromised, return false
      if (window.jeetBlockSecurity && window.jeetBlockSecurity.isExtensionCompromised()) {
        return { licensed: false, error: 'Security compromised' };
      }
      
      const deviceId = await this.getDeviceId();
      
      // Use security trap function for device ID validation
      if (window.jeetBlockSecurity && !window.jeetBlockSecurity.validateDeviceId(deviceId)) {
        return { licensed: false, error: 'Device validation failed' };
      }
      
      const response = await fetch(`${this.licenseCheckUrl}/check-license`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        mode: 'cors',
        credentials: 'omit',
        body: JSON.stringify({
          device_id: deviceId,
          product_tier: 'JeetBlock Basic'
        })
      });

      if (!response.ok) {
        const errorText = await response.text();
        console.error('JeetBlock: License check failed:', response.status, errorText);
        throw new Error(`HTTP error! status: ${response.status}, body: ${errorText}`);
      }

      const data = await response.json();
      console.log('JeetBlock: License check response:', data);
      
      // Use security trap function for license validation
      if (window.jeetBlockSecurity && !window.jeetBlockSecurity.validateLicense(data)) {
        return { licensed: false, error: 'License validation failed' };
      }
      
      this.isLicensed = data.licensed || false;
      this.licenseKey = data.licenseKey || null;
      
      // Store license status with security validation
      if (window.jeetBlockSecurity && !window.jeetBlockSecurity.isExtensionCompromised()) {
        // Store in chrome.storage.local for content script access
        await chrome.storage.local.set({
          'jeetblock_licensed': this.isLicensed,
          'jeetblock_license_key': this.licenseKey,
          'jeetblock_last_validation': Date.now(),
          'jeetblock_server_validated': this.isLicensed // Required for Basic version security check
        });
        
        // Also store in localStorage as backup
        localStorage.setItem('jeetblock_licensed', this.isLicensed.toString());
        if (this.licenseKey) {
          localStorage.setItem('jeetblock_license_key', this.licenseKey);
        }
        localStorage.setItem('jeetblock_last_validation', Date.now().toString());
      }
      
      return {
        licensed: this.isLicensed,
        licenseKey: this.licenseKey
      };
    } catch (error) {
      console.error('Error checking license:', error);
      return {
        licensed: false,
        error: error.message
      };
    }
  }

  // Get stored license status
  async getStoredLicenseStatus() {
    try {
      // Use chrome.storage.local for better persistence
      const result = await chrome.storage.local.get({
        'jeetblock_licensed': false,
        'jeetblock_dev_mode': false,
        'jeetblock_license_key': null
      });
      
      console.log('JeetBlock: Raw chrome.storage.local values:', result);
      
      this.isLicensed = result.jeetblock_licensed;
      this.licenseKey = result.jeetblock_license_key;
      this.devMode = result.jeetblock_dev_mode;
      
      const status = {
        licensed: this.isLicensed,
        licenseKey: this.licenseKey,
        devMode: this.devMode
      };
      
      console.log('JeetBlock: Parsed license status:', status);
      return status;
    } catch (error) {
      console.error('JeetBlock: Error getting stored license status:', error);
      // Fallback to localStorage if chrome.storage fails
      const licensed = localStorage.getItem('jeetblock_licensed') === 'true';
      const licenseKey = localStorage.getItem('jeetblock_license_key');
      const devMode = localStorage.getItem('jeetblock_dev_mode') === 'true';
      
      this.isLicensed = licensed;
      this.licenseKey = licenseKey;
      this.devMode = devMode;
      
      return {
        licensed: this.isLicensed,
        licenseKey: this.licenseKey,
        devMode: this.devMode
      };
    }
  }

  // Enable dev mode
  async enableDevMode() {
    console.log('JeetBlock: Enabling dev mode...');
    
    this.devMode = true;
    this.isLicensed = true;
    this.licenseKey = 'DEV-MODE-BYPASS';
    
    try {
      // Store dev mode status in chrome.storage.local
      await chrome.storage.local.set({
        'jeetblock_dev_mode': true,
        'jeetblock_licensed': true,
        'jeetblock_license_key': 'DEV-MODE-BYPASS',
        'jeetblock_last_validation': Date.now(),
        'jeetblock_server_validated': true // Required for Basic version security check
      });
      
      console.log('JeetBlock: Dev mode enabled, chrome.storage.local updated');
      
      // Also store in localStorage as backup
      localStorage.setItem('jeetblock_dev_mode', 'true');
      localStorage.setItem('jeetblock_licensed', 'true');
      localStorage.setItem('jeetblock_license_key', 'DEV-MODE-BYPASS');
      localStorage.setItem('jeetblock_last_validation', Date.now().toString());
      
      console.log('JeetBlock: Dev mode stored in both chrome.storage.local and localStorage');
    } catch (error) {
      console.error('JeetBlock: Error storing dev mode status:', error);
      // Fallback to localStorage only
      localStorage.setItem('jeetblock_dev_mode', 'true');
      localStorage.setItem('jeetblock_licensed', 'true');
      localStorage.setItem('jeetblock_license_key', 'DEV-MODE-BYPASS');
      localStorage.setItem('jeetblock_last_validation', Date.now().toString());
    }
  }

  // Disable dev mode
  disableDevMode() {
    this.devMode = false;
    this.isLicensed = false;
    this.licenseKey = null;
    
    // Clear dev mode status
    localStorage.removeItem('jeetblock_dev_mode');
    localStorage.removeItem('jeetblock_licensed');
    localStorage.removeItem('jeetblock_license_key');
    localStorage.removeItem('jeetblock_last_validation');
    
    console.log('JeetBlock: Dev mode disabled - normal licensing restored');
  }

  // Check if dev mode is enabled
  isDevModeEnabled() {
    return this.devMode || localStorage.getItem('jeetblock_dev_mode') === 'true';
  }

  // Start polling for license status
  startPolling(callback) {
    this.pollingStartTime = Date.now();
    
    this.pollingInterval = setInterval(async () => {
      const elapsed = Date.now() - this.pollingStartTime;
      
      if (elapsed >= this.maxPollingTime) {
        this.stopPolling();
        callback({ licensed: false, timeout: true });
        return;
      }
      
      const result = await this.checkLicense();
      if (result.licensed) {
        this.stopPolling();
        callback({ licensed: true, licenseKey: result.licenseKey });
      }
    }, 10000); // Poll every 10 seconds
  }

  // Stop polling
  stopPolling() {
    if (this.pollingInterval) {
      clearInterval(this.pollingInterval);
      this.pollingInterval = null;
    }
    if (this.pollingTimeout) {
      clearTimeout(this.pollingTimeout);
      this.pollingTimeout = null;
    }
  }

  // Open payment URLs
  openCryptoPayment() {
    this.getDeviceId().then(deviceId => {
      chrome.tabs.create({
        url: `https://site.jeetblock.org/basic-crypto?deviceId=${encodeURIComponent(deviceId)}`
      });
    });
  }

  openCardPayment() {
    this.getDeviceId().then(deviceId => {
      chrome.tabs.create({
        url: `https://site.jeetblock.org/basic-card?deviceId=${encodeURIComponent(deviceId)}`
      });
    });
  }

  // Update authentication endpoint
  async updateAuthEndpoint(newUrl) {
    this.licenseCheckUrl = newUrl;
    try {
      await chrome.storage.local.set({ 'jeetblock_auth_endpoint': newUrl });
      console.log('JeetBlock: Authentication endpoint updated to:', newUrl);
    } catch (error) {
      console.error('Error saving endpoint:', error);
    }
  }

  // Clear license data (for testing)
  clearLicenseData() {
    localStorage.removeItem(this.storageKey);
    localStorage.removeItem('jeetblock_licensed');
    localStorage.removeItem('jeetblock_license_key');
    this.deviceId = null;
    this.isLicensed = false;
    this.licenseKey = null;
  }
}

// Export for use in other files
window.JeetBlockLicensing = JeetBlockLicensing; 