// Popup Script for Social Media Harmonizer
class PopupManager {
  constructor() {
    this.defaultSettings = {
      filterMode: 'collapse',
      isEnabled: false, // Start disabled by default
      customFilters: [],
      filterIndian: true,  // Only Indian content filter enabled
      filterMuslim: false, // Disabled and locked
      filterJewish: false, // Disabled and locked
      filterLGBTQIA: false, // Disabled and locked
      filterUkraine: false, // Disabled and locked
      filterPalestine: false, // Disabled and locked
      // Platform-specific filtering
      filterFacebook: true,  // Enabled
      filterTwitter: true,   // Enabled
      filterInstagram: true,  // Enabled
      filterLinkedIn: true,  // Enabled
      filterTikTok: false,   // Disabled and locked
      filterReddit: false,   // Disabled and locked
      filterYouTube: false   // Disabled and locked
    };
    this.settings = { ...this.defaultSettings };
    this.booleanSettingKeys = [
      'isEnabled',
      'filterIndian',
      'filterMuslim',
      'filterJewish',
      'filterLGBTQIA',
      'filterUkraine',
      'filterPalestine',
      'filterFacebook',
      'filterTwitter',
      'filterInstagram',
      'filterLinkedIn',
      'filterTikTok',
      'filterReddit',
      'filterYouTube'
    ];
    this.licensing = new JeetBlockLicensing();
    this.isLicensed = false;
    this.init();
  }

  parseBoolean(value, defaultValue) {
    if (typeof value === 'boolean') {
      return value;
    }

    if (typeof value === 'string') {
      const normalized = value.trim().toLowerCase();
      if (normalized === 'true' || normalized === '1') {
        return true;
      }
      if (normalized === 'false' || normalized === '0' || normalized === '') {
        return false;
      }
    }

    if (typeof value === 'number') {
      if (value === 1) {
        return true;
      }
      if (value === 0) {
        return false;
      }
    }

    if (value === null || value === undefined) {
      return defaultValue;
    }

    return Boolean(value);
  }

  // Theme detection and application
  applyTheme() {
    // Check if user has a stored theme preference
    const storedTheme = localStorage.getItem('jeetblock-theme');
    
    if (storedTheme) {
      // Use stored preference
      this.setTheme(storedTheme);
    } else {
      // Auto-detect system theme
      const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
      const theme = prefersDark ? 'dark' : 'light';
      this.setTheme(theme);
    }
    
    // Listen for system theme changes
    window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
      if (!localStorage.getItem('jeetblock-theme')) {
        // Only auto-switch if user hasn't set a manual preference
        this.setTheme(e.matches ? 'dark' : 'light');
      }
    });
  }

  setTheme(theme) {
    const body = document.body;
    const container = document.querySelector('.container');
    
    if (theme === 'dark') {
      body.classList.add('dark-theme');
      body.classList.remove('light-theme');
      if (container) {
        container.classList.add('dark-theme');
        container.classList.remove('light-theme');
      }
    } else {
      body.classList.add('light-theme');
      body.classList.remove('dark-theme');
      if (container) {
        container.classList.add('light-theme');
        container.classList.remove('dark-theme');
      }
    }
    
    // Store theme preference
    localStorage.setItem('jeetblock-theme', theme);
  }

  toggleTheme() {
    const currentTheme = localStorage.getItem('jeetblock-theme') || 
                        (window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light');
    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
    this.setTheme(newTheme);
  }

  updateThemeButton() {
    const themeToggle = document.getElementById('themeToggle');
    if (themeToggle) {
      const currentTheme = localStorage.getItem('jeetblock-theme') || 
                          (window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light');
      themeToggle.textContent = currentTheme === 'dark' ? '☀️' : '🌙';
      themeToggle.title = currentTheme === 'dark' ? 'Switch to light mode' : 'Switch to dark mode';
    }
  }

  async updateJeetblockId() {
    const idDisplay = document.getElementById('jeetblockIdDisplay');
    if (!idDisplay) {
      console.warn('JeetBlock: ID display element not found');
      return;
    }

    try {
      // Try to get device ID from licensing object
      if (this.licensing && typeof this.licensing.getDeviceId === 'function') {
        const deviceId = await this.licensing.getDeviceId();
        if (deviceId) {
          idDisplay.textContent = deviceId;
          return;
        }
      }
      
      // Fallback: Try to get directly from storage
      let deviceId = null;
      
      // Check localStorage first with version-specific key
      try {
        deviceId = localStorage.getItem('jeetblock_device_id_basic');
      } catch (e) {
        console.warn('JeetBlock: Could not access localStorage:', e);
      }
      
      // Check chrome.storage.local as backup with version-specific key
      if (!deviceId && typeof chrome !== 'undefined' && chrome.storage) {
        try {
          const result = await chrome.storage.local.get(['jeetblock_device_id_basic']);
          if (result && result.jeetblock_device_id_basic) {
            deviceId = result.jeetblock_device_id_basic;
          }
        } catch (e) {
          console.warn('JeetBlock: Could not access chrome.storage.local:', e);
        }
      }
      
      if (deviceId) {
        idDisplay.textContent = deviceId;
      } else {
        console.error('JeetBlock: Device ID not found in storage');
        idDisplay.textContent = 'ID not found';
      }
    } catch (error) {
      console.error('JeetBlock: Error loading JEETBLOCK ID:', error);
      idDisplay.textContent = 'Error loading ID';
    }
  }

  async init() {
    console.log('JeetBlock: Popup initializing...');
    
    // Security check - if compromised, don't initialize
    if (window.jeetBlockSecurity && window.jeetBlockSecurity.isExtensionCompromised()) {
      this.showSecurityError();
      return;
    }
    
    // Apply theme immediately
    this.applyTheme();
    this.updateThemeButton();
    
    await this.loadSettings();
    await this.checkLicensing();
    this.setupEventListeners();
    this.updateUI();
    
    // Update JEETBLOCK ID after a small delay to ensure licensing is ready
    setTimeout(() => {
      this.updateJeetblockId();
    }, 100);
    
    console.log('JeetBlock: Popup initialized successfully');
    
    // Send current settings to content script
    this.updateContentScript();
    
    // Also force reload settings in content script
    try {
      await chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
        if (tabs[0]) {
          chrome.tabs.sendMessage(tabs[0].id, {action: 'reloadSettings'});
        }
      });
    } catch (error) {
      console.error('JeetBlock: Error reloading settings in content script:', error);
    }
    
    // Update icon based on current state
    try {
      await chrome.runtime.sendMessage({
        action: 'updateIcon',
        enabled: this.settings.isEnabled
      });
    } catch (error) {
      console.error('JeetBlock: Error updating icon on init:', error);
    }
    
    // Also update UI when popup becomes visible (in case settings changed)
    document.addEventListener('visibilitychange', () => {
      if (!document.hidden) {
        this.loadSettings().then(() => this.updateUI());
      }
    });
  }

  async loadSettings() {
    const result = await chrome.storage.sync.get(null);

    console.log('JeetBlock: Raw storage result:', result);

    // Start with defaults
    this.settings = { ...this.defaultSettings };

    // Apply stored primitive settings if defined
    if (typeof result.filterMode === 'string') {
      this.settings.filterMode = result.filterMode;
    }

    ['customFilters'].forEach(key => {
      if (Array.isArray(result[key])) {
        this.settings[key] = result[key];
      }
    });

    this.booleanSettingKeys.forEach(key => {
      if (Object.prototype.hasOwnProperty.call(result, key)) {
        this.settings[key] = this.parseBoolean(result[key], this.defaultSettings[key]);
      }
    });

    console.log('JeetBlock: Settings loaded in popup:', this.settings);
    console.log('JeetBlock: isEnabled value:', this.settings.isEnabled, 'type:', typeof this.settings.isEnabled);
  }

  async checkLicensing() {
    try {
      // First check stored license status
      const storedStatus = await this.licensing.getStoredLicenseStatus();
      this.isLicensed = storedStatus.licensed;
      
      console.log('JeetBlock: Popup checking license status:', storedStatus);
      
      if (this.isLicensed) {
        // Don't modify isEnabled - let it keep the user's preference
        this.hidePaywall();
        console.log('JeetBlock: Popup - licensed, hiding paywall');
        return; // Already licensed
      }
      
      // Check with server
      const result = await this.licensing.checkLicense();
      this.isLicensed = result.licensed;
      
      if (this.isLicensed) {
        // Don't modify isEnabled - let it keep the user's preference
        this.hidePaywall();
        console.log('JeetBlock: Popup - server license verified, hiding paywall');
      } else {
        this.settings.isEnabled = false; // Disable extension if not licensed
        this.showPaywall();
        console.log('JeetBlock: Popup - not licensed, showing paywall');
      }
    } catch (error) {
      console.error('Error checking license:', error);
      this.settings.isEnabled = false; // Disable extension on error
      this.showPaywall();
    }
  }

  // Update authentication endpoint (for future use)
  async updateAuthEndpoint(newUrl) {
    try {
      await this.licensing.updateAuthEndpoint(newUrl);
      
      // Also update background script
      chrome.runtime.sendMessage({
        action: 'updateAuthEndpoint',
        url: newUrl
      });
      
      this.showNotification('Authentication endpoint updated successfully!', 'success');
    } catch (error) {
      console.error('Error updating endpoint:', error);
      this.showNotification('Failed to update endpoint', 'error');
    }
  }


  showPaywall() {
    const paywallModal = document.getElementById('paywallModal');
    const deviceIdDisplay = document.getElementById('deviceIdDisplay');
    
    // Show device ID
    this.licensing.getDeviceId().then(deviceId => {
      deviceIdDisplay.textContent = deviceId;
    });
    
    // Show modal
    paywallModal.style.display = 'flex';
    
    // Disable main UI
    document.querySelector('.container').style.pointerEvents = 'none';
    document.querySelector('.container').style.opacity = '0.3';
  }

  hidePaywall() {
    const paywallModal = document.getElementById('paywallModal');
    paywallModal.style.display = 'none';
    
    // Re-enable main UI
    document.querySelector('.container').style.pointerEvents = 'auto';
    document.querySelector('.container').style.opacity = '1';
  }

  setupPaywallEventListeners() {
    // Crypto payment button
    document.getElementById('cryptoPaymentBtn').addEventListener('click', () => {
      this.handlePayment('crypto');
    });
    
    // Card payment button
    document.getElementById('cardPaymentBtn').addEventListener('click', () => {
      this.handlePayment('card');
    });
  }

  handlePayment(paymentType) {
    const paymentStatus = document.getElementById('paymentStatus');
    const statusText = document.getElementById('statusText');
    const paywallButtons = document.querySelector('.paywall-buttons');
    
    // Show payment status
    paymentStatus.style.display = 'block';
    paywallButtons.style.display = 'none';
    
    // Open payment URL
    if (paymentType === 'crypto') {
      this.licensing.openCryptoPayment();
    } else {
      this.licensing.openCardPayment();
    }
    
    // Start polling for license
    this.licensing.startPolling((result) => {
      if (result.licensed) {
        this.isLicensed = true;
        // Don't automatically enable - let user decide
        // this.settings.isEnabled = true;
        this.hidePaywall();
        this.showNotification('License activated! Welcome to JeetBlock!', 'success');
        
        // Update UI to show current state
        this.updateUI();
        
        // Reinitialize content script with new license status
        this.reinitializeContentScript();
      } else if (result.timeout) {
        statusText.textContent = 'Payment not detected yet. Please try again.';
        paymentStatus.classList.add('error');
        paywallButtons.style.display = 'flex';
        paymentStatus.style.display = 'none';
      }
    });
  }

  async saveSettings() {
    console.log('JeetBlock: Saving settings:', this.settings);

    // Persist all settings as simple JSON-friendly values
    const payload = {};

    Object.entries(this.settings).forEach(([key, value]) => {
      if (this.booleanSettingKeys.includes(key)) {
        payload[key] = this.parseBoolean(value, this.defaultSettings[key]);
      } else if (Array.isArray(value)) {
        payload[key] = value.slice();
      } else if (value !== undefined) {
        payload[key] = value;
      }
    });

    await chrome.storage.sync.set(payload);
    
    this.updateContentScript();
    
    // Update icon based on enabled state
    try {
      await chrome.runtime.sendMessage({
        action: 'updateIcon',
        enabled: this.settings.isEnabled
      });
    } catch (error) {
      console.error('JeetBlock: Error updating icon:', error);
    }
  }

  setupEventListeners() {
    // Setup paywall event listeners
    this.setupPaywallEventListeners();
    
    // Theme toggle
    const themeToggle = document.getElementById('themeToggle');
    if (themeToggle) {
      themeToggle.addEventListener('click', () => {
        this.toggleTheme();
        this.updateThemeButton();
      });
    }

    // Enable/Disable toggle
    document.getElementById('enableToggle').addEventListener('change', (e) => {
      console.log('JeetBlock: Enable toggle changed to:', e.target.checked);
      this.settings.isEnabled = e.target.checked;
      this.saveSettings();
    });

    // Filter category toggles - removed (Indian filter always on, others locked)
    
    // Platform-specific toggles - enabled platforms only
    document.getElementById('filterFacebookToggle').addEventListener('change', (e) => {
      this.settings.filterFacebook = e.target.checked;
      this.saveSettings();
    });

    document.getElementById('filterTwitterToggle').addEventListener('change', (e) => {
      this.settings.filterTwitter = e.target.checked;
      this.saveSettings();
    });

    document.getElementById('filterInstagramToggle').addEventListener('change', (e) => {
      this.settings.filterInstagram = e.target.checked;
      this.saveSettings();
    });

    document.getElementById('filterLinkedInToggle').addEventListener('change', (e) => {
      this.settings.filterLinkedIn = e.target.checked;
      this.saveSettings();
    });

    // Filter mode radio buttons
    document.querySelectorAll('input[name="filterMode"]').forEach(radio => {
      radio.addEventListener('change', (e) => {
        this.settings.filterMode = e.target.value;
        this.saveSettings();
      });
    });


  }


  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  updateUI() {
    console.log('JeetBlock: Updating UI with settings:', this.settings);
    
    // Update toggles
    document.getElementById('enableToggle').checked = this.settings.isEnabled;
    
    // Indian filter is always on (no toggle needed)
    // Other filters are locked (no toggles needed)
    
    // Update platform toggles
    document.getElementById('filterFacebookToggle').checked = this.settings.filterFacebook;
    document.getElementById('filterTwitterToggle').checked = this.settings.filterTwitter;
    document.getElementById('filterInstagramToggle').checked = this.settings.filterInstagram;
    document.getElementById('filterLinkedInToggle').checked = this.settings.filterLinkedIn;
    
    // Locked platform toggles removed (no toggles needed)
    
    // Update radio buttons
    document.querySelectorAll('input[name="filterMode"]').forEach(radio => {
      radio.checked = radio.value === this.settings.filterMode;
    });
    
    console.log('JeetBlock: UI updated - filterMuslim:', this.settings.filterMuslim, 'filterJewish:', this.settings.filterJewish);
  }

  isSupportedSite(url) {
    const supportedDomains = [
      'facebook.com',
      'twitter.com',
      'x.com',
      'instagram.com',
      'linkedin.com',
      'tiktok.com'
    ];
    
    return supportedDomains.some(domain => url.includes(domain));
  }

  async updateContentScript() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && this.isSupportedSite(tab.url)) {
        await chrome.tabs.sendMessage(tab.id, {
          action: 'updateSettings',
          settings: this.settings
        });
      }
    } catch (error) {
      // Content script not available
      console.log('Content script not available');
    }
  }

  async reinitializeContentScript() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && this.isSupportedSite(tab.url)) {
        console.log('JeetBlock: Attempting to reinitialize content script...');
        
        // First try to force enable the extension
        try {
          const response = await chrome.tabs.sendMessage(tab.id, {
            action: 'forceEnable'
          });
          
          if (response && response.enabled) {
            console.log('JeetBlock: Content script force enabled successfully');
            // Also update settings after enabling
            this.updateContentScript();
            return;
          }
        } catch (error) {
          console.log('JeetBlock: Force enable failed, trying reinitialize...');
        }
        
        // If force enable fails, try reinitialize
        try {
          const response = await chrome.tabs.sendMessage(tab.id, {
            action: 'reinitialize'
          });
          
          if (response && response.licensed) {
            console.log('JeetBlock: Content script reinitialized successfully');
            // Also update settings after reinitialization
            this.updateContentScript();
          } else {
            console.log('JeetBlock: Content script reinitialization failed');
          }
        } catch (error) {
          console.log('JeetBlock: Reinitialize failed, will retry...');
          // Retry after a delay
          setTimeout(async () => {
            try {
              await this.updateContentScript();
              console.log('JeetBlock: Settings updated after retry');
            } catch (retryError) {
              console.error('JeetBlock: Retry failed:', retryError);
            }
          }, 1000);
        }
      }
    } catch (error) {
      console.error('Error reinitializing content script:', error);
    }
  }



  showSecurityError() {
    // Show generic error message to user
    const container = document.querySelector('.container');
    if (container) {
      container.innerHTML = `
        <div style="text-align: center; padding: 20px; color: #666;">
          <h3>Extension temporarily unavailable</h3>
          <p>Please try refreshing the page or reinstalling the extension.</p>
        </div>
      `;
    }
  }

  showNotification(message, type = 'info') {
    // Security check - if compromised, don't show notifications
    if (window.jeetBlockSecurity && window.jeetBlockSecurity.isExtensionCompromised()) {
      return;
    }
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      padding: 12px 16px;
      border-radius: 6px;
      color: white;
      font-size: 14px;
      font-weight: 500;
      z-index: 1000;
      animation: slideIn 0.3s ease-out;
    `;
    
    // Set background color based on type
    if (type === 'success') {
      notification.style.backgroundColor = '#28a745';
    } else if (type === 'error') {
      notification.style.backgroundColor = '#dc3545';
    } else {
      notification.style.backgroundColor = '#17a2b8';
    }
    
    // Add animation styles
    const style = document.createElement('style');
    style.textContent = `
      @keyframes slideIn {
        from {
          transform: translateX(100%);
          opacity: 0;
        }
        to {
          transform: translateX(0);
          opacity: 1;
        }
      }
    `;
    document.head.appendChild(style);
    
    document.body.appendChild(notification);
    
    // Remove notification after 3 seconds
    setTimeout(() => {
      notification.style.animation = 'slideOut 0.3s ease-in';
      setTimeout(() => {
        if (notification.parentNode) {
          notification.parentNode.removeChild(notification);
        }
      }, 300);
    }, 3000);
  }
}

// Initialize popup when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  new PopupManager();
}); 